<?php
/*
| -------------------------------------------------------------
| AUTHOR:           INILABS TEAM
| -------------------------------------------------------------
| EMAIL:            info@inilabs.net or labs.ini@gmail.com
| -------------------------------------------------------------
| WEBSITE:          http://inilabs.net
| -------------------------------------------------------------
*/
namespace Modules\Bank\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Modules\Bank\Models\Bank;
use Modules\Bank\Repositories\BankRepository;
use Modules\Bank\Requests\BankRequest;
use Modules\Deposit\Models\Deposit;
use Yajra\Datatables\Facades\Datatables;
use Alert, Cache;

class BankController extends Controller
{
    protected $bankRepo;

    function __construct(BankRepository $bank)
    {
        $this->bankRepo = $bank;
    }

    public function index()
    {
        return view("Bank::bankList");
    }

    public function bankDataTable()
    {
        return DataTables::of($this->bankRepo->with(['bankPayments.payment', 'deposits'])->findAll(['*']))
            ->addColumn('actions', function ($bank){
                return $bank->action_buttons;
            })
            ->editColumn('ini_balance', function ($bank){
                return number_format($bank->ini_balance, 2);
            })
            ->addColumn('balance', function ($bank){
                $bankAmount = 0;
                $current_balance = $bank->ini_balance + $bank->deposits->sum('amount');
                foreach ($bank->bankPayments as $bankPayment) {
                    $bankAmount += $bankPayment->payment->paid_amount;
                }
                $current_balance += $bankAmount;
                return number_format($current_balance, 2);
            })
            ->make(true);
    }

    public function add()
    {
        return view("Bank::bankCreate");
    }

    public function create(BankRequest $request)
    {
        $input = $request->all();
        $input['user_id'] = auth()->user()->id;
        $result = $this->bankRepo->create($input);

        if ($result) {
            $this->removeCached();
            Alert::success(trans("Bank::bank.alert.update_success"), 'success');
            return redirect()->route('bank.list');
        }
    }

    public function edit(Bank $bank)
    {
        return view("Bank::bankEdit")->withBank($bank);
    }

    public function update(Bank $bank, BankRequest $request)
    {
        $result = $this->bankRepo->update($bank, $request->all());

        if ($result) {
            $this->removeCached();
            Alert::success(trans("Bank::bank.alert.update_success"), 'success');
            return redirect()->route('bank.list');
        }
    }

    public function view($bank)
    {
        $bank = Bank::with('deposits', 'bankPayments.payment')->findOrFail($bank);
        return $bank != null ? view("Bank::bankView")->withBank($bank) :  abort(404);
    }

    public function status(Bank $bank)
    {
        $data = [];
        if ($bank->status==1) {
            $data['status'] = 0;
        } else {
            $data['status'] = 1;
        }
        $this->bankRepo->update($bank->id, $data);
        $this->removeCached();
        Alert::success(trans("Bank::bank.alert.status_success"), 'success');
        return redirect()->route('bank.list');
    }

    public function delete(Bank $bank)
    {
        $this->bankRepo->delete($bank->id);

        $this->removeCached();
        Alert::success(trans("Bank::bank.alert.delete_success"), 'success');
        return redirect()->route('bank.list');
    }

    public function removeCached()
    {
        Cache::forget('banks');
        Cache::forget('bankBalance');
    }


}
