<?php
/*
| -------------------------------------------------------------
| AUTHOR:           INILABS TEAM
| -------------------------------------------------------------
| EMAIL:            info@inilabs.net or labs.ini@gmail.com
| -------------------------------------------------------------
| WEBSITE:          http://inilabs.net
| -------------------------------------------------------------
*/
namespace Modules\Deposit\Controllers;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Modules\Bank\Models\Bank;
use Modules\Deposit\Models\Deposit;
use Modules\Deposit\Repositories\DepositRepository;
use Modules\Deposit\Requests\DepositRequest;
use Yajra\Datatables\Facades\Datatables;
use File, Alert, Image, Cache;

class DepositController extends Controller
{
    protected $depositRepo;

    function __construct(DepositRepository $deposit)
    {
        $this->depositRepo = $deposit;
    }

    public function index()
    {
        return view("Deposit::depositList");
    }

    public function depositDataTable()
    {
        return DataTables::of($this->depositRepo->findAll(['*']))
            ->addColumn('actions', function ($deposit){
                return $deposit->action_buttons;
            })
            ->editColumn('date', function ($deposit){
                return Carbon::parse($deposit->date)->format('Y-m-d');
            })
            ->addColumn('balance', function ($deposit){
                return number_format($deposit->bank->ini_balance+$deposit->amount, 2);
            })
            ->editColumn('amount', function ($deposit){
                return number_format($deposit->amount, 2);
            })
            ->editColumn('attach', function ($deposit){
                if ($deposit->attach) {
                    return '<a target="_blank" class="btn btn-xs btn-info" href="'.url("public/assets/img/deposit/".$deposit->attach).'" download>'."Download"."</a>";
                }
                return '';
            })
            ->editColumn('bank_id', function ($deposit){
                return $deposit->bank->bank_title;
            })
            ->make(true);
    }

    public function add()
    {
        $banks = Cache::remember('banks',60, function () {
            return Bank::where('status', 1)->pluck('bank_title', 'id')->all();
        });
        return view("Deposit::depositCreate")->with('banks', $banks);
    }

    public function create(DepositRequest $request)
    {
        $request['user_id'] = auth()->user()->id;
        if ($request->hasFile('attachment')) {
            $attachName = str_random(32);
            $ext = $request->file('attachment')->extension();
            if(!is_dir(base_path(config('core.assets_path.deposit')))) {
                File::makeDirectory(base_path(config('core.assets_path.deposit')));
            }
            Image::make($request->file('attachment'))->save(config('core.assets_path.deposit').$attachName.'.'.$ext);
            $request['attach'] = $attachName.'.'.$ext;
        }
        $result = $this->depositRepo->create($request->except('attachment'));

        if ($result) {
            $this->removeCached();
            Alert::success(trans("Deposit::deposit.alert.update_success"), 'success');
            return redirect()->route('deposit.list');
        }
    }

    public function edit(Deposit $deposit)
    {
        $banks = Cache::remember('banks',60, function () {
            return Bank::where('status', 1)->pluck('bank_title', 'id')->all();
        });
        $deposit['date'] = Carbon::parse($deposit['date'])->format('Y-m-d');
        return view("Deposit::depositEdit")->withDeposit($deposit)->with('banks', $banks);
    }

    public function update(Deposit $deposit, DepositRequest $request)
    {
        if ($request->hasFile('attachment')) {
            $attachName = str_random(32);
            $ext = $request->file('attachment')->extension();
            if(!is_dir(base_path(config('core.assets_path.deposit')))) {
                File::makeDirectory(base_path(config('core.assets_path.deposit')));
            }
            Image::make($request->file('attachment'))->save(config('core.assets_path.deposit').$attachName.'.'.$ext);
            $request['attach'] = $attachName.'.'.$ext;
        }
        $result = $this->depositRepo->update($deposit, $request->except('attachment'));

        if ($result) {
            $this->removeCached();
            Alert::success(trans("Deposit::deposit.alert.update_success"), 'success');
            return redirect()->route('deposit.list');
        }
    }

    public function view(Deposit $deposit)
    {
        return $deposit != null ? view("Deposit::depositView")->withDeposit($deposit) :  abort(404);
    }

    public function status(Deposit $deposit)
    {
        $data = [];
        if ($deposit->status==1) {
            $data['status'] = 0;
        } else {
            $data['status'] = 1;
        }
        $this->depositRepo->update($deposit->id, $data);
        $this->removeCached();
        Alert::success(trans("Deposit::deposit.alert.status_success"), 'success');
        return redirect()->route('deposit.list');
    }

    public function delete(Deposit $deposit)
    {
        $this->depositRepo->delete($deposit->id);
        $this->removeCached();
        Alert::success(trans("Deposit::deposit.alert.delete_success"), 'success');
        return redirect()->route('deposit.list');
    }

    public function removeCached()
    {
        Cache::forget('bankBalance');
    }


}
