<?php
/*
| -------------------------------------------------------------
| AUTHOR:           INILABS TEAM
| -------------------------------------------------------------
| EMAIL:            info@inilabs.net or labs.ini@gmail.com
| -------------------------------------------------------------
| WEBSITE:          http://inilabs.net
| -------------------------------------------------------------
*/
namespace Modules\Expense\Controllers;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Core\iSite;
use Illuminate\Support\Facades\Auth;
use Modules\Expense\Models\Expense;
use Modules\Expense\Repositories\ExpenseRepository;
use Modules\Expense\Requests\ExpenseRequest;
use Modules\ExpenseType\Models\ExpenseType;
use Modules\Iauth\Models\Access\User\User;
use Modules\PaymentType\Models\PaymentType;
use Yajra\Datatables\Facades\Datatables;
use File, Alert, Image, Cache;
use DB;
class ExpenseController extends Controller
{
    protected $expenseRepo;

    function __construct(ExpenseRepository $expense)
    {
        $this->expenseRepo = $expense;
    }

    public function index()
    {
        return view("Expense::expenseList");
    }

    public function expenseDataTable()
    {
        $expenses = $this->expenseRepo->orderBy('created_at', 'desc')->findAll();

        return DataTables::of($expenses)
            ->addColumn('purchased', function ($expense){
                return $expense->purchased->name;
            })
            ->editColumn('purchased_date', function ($expense){
                return $expense->purchased_date ? with(new Carbon($expense->purchased_date))->format('jS F Y') : '';
            })
            ->editColumn('expense_type_id', function ($expense){
                return $expense->expenseType->name;
            })
            ->editColumn('purchased_amount', function ($expense){
                return isite()->siteCurrencySymbol().number_format($expense->purchased_amount, 2);
            })
            ->addColumn('actions', function ($expense){
                return $expense->action_buttons;
            })
            ->make(true);
    }

    public function add()
    {
        $users = Cache::remember('users',60, function () {
            return User::pluck('username', 'id')->toArray();
        });
        $expense_types = Cache::remember('expense_types',60, function () {
            return ExpenseType::where('status', 1)->pluck('name', 'id')->all();
        });
        $payment_types = Cache::remember('payment_types',60, function () {
            return PaymentType::where('status', 1)->pluck('name', 'id')->toArray();
        });

        return view("Expense::expenseCreate")->withUsers($users)->withExpenseTypes($expense_types)->withPaymentTypes($payment_types);
    }

    public function create(ExpenseRequest $request)
    {
        $request->all();
        $date = new \DateTime($request->purchased_date);
        $request['purchased_date'] = $date->format('Y-m-d');
        $request['user_id'] = auth()->user()->id;
        if ($request->hasFile('attachment')) {
            $attachName = str_random(32);
            $ext = $request->file('attachment')->extension();
            if(!is_dir(base_path(config('core.assets_path.expense_image')))) {
                File::makeDirectory(base_path(config('core.assets_path.expense_image')));
            }
            Image::make($request->file('attachment'))->save(config('core.assets_path.expense_image').$attachName.'.'.$ext);
            $request['attach'] = $attachName.'.'.$ext;
        }
        $result = $this->expenseRepo->create($request->except('attachment'));
        if ($result) {
            $this->removeCached();
            Alert::success(trans("Expense::expense.alert.update_success"), 'success');
            return redirect()->route('expense.list');
        }
    }

    public function edit(Expense $expense)
    {
        $users = Cache::remember('users',60, function () {
            return User::pluck('username', 'id')->toArray();
        });
        $expense_types = Cache::remember('expense_types',60, function () {
            return ExpenseType::where('status', 1)->pluck('name', 'id')->all();
        });
        $payment_types = Cache::remember('payment_types',60, function () {
            return PaymentType::where('status', 1)->pluck('name', 'id')->toArray();
        });
        return view("Expense::expenseEdit")->withExpense($expense)->withUsers($users)->withExpenseTypes($expense_types)->withPaymentTypes($payment_types);
    }

    public function update(Expense $expense, ExpenseRequest $request)
    {
        $date = new \DateTime($request->purchased_date);
        $request['purchased_date'] = $date->format('Y-m-d');
        if ($request->hasFile('attachment')) {
            $attachName = str_random(32);
            $ext = $request->file('attachment')->extension();
            if(!is_dir(base_path(config('core.assets_path.expense_image')))) {
                File::makeDirectory(base_path(config('core.assets_path.expense_image')));
            }
            Image::make($request->file('attachment'))->save(config('core.assets_path.expense_image').$attachName.'.'.$ext);
            $request['attach'] = $attachName.'.'.$ext;
        }
        $result = $this->expenseRepo->update($expense, $request->except('attachment'));
        if ($result) {
            $this->removeCached();
            Alert::success(trans("Expense::expense.alert.update_success"), 'success');
            return redirect()->route('expense.list');
        }
    }

    public function view(Expense $expense)
    {
        return $expense != null ? view("Expense::expenseView")->withExpense($expense) :  abort(404);
    }

    public function status(Expense $expense)
    {
        $data = [];

        if ($expense->status==1) {
            $data['status'] = 0;
        } else {
            $data['status'] = 1;
        }
        $this->expenseRepo->update($expense->id, $data);
        $this->removeCached();
        Alert::success(trans("Expense::expense.alert.status_success"), 'success');

        return redirect()->route('expense.list');
    }

    public function delete(Expense $expense)
    {
        if($expense->user_id == auth()->user()->id || Auth::hasRole('Admin')) {
            $this->expenseRepo->delete($expense->id);
            $this->removeCached();
            Alert::success(trans("Expense::expense.alert.delete_success"), 'success');
            return redirect()->route('expense.list');
        } else {
            $this->removeCached();
            Alert::error(trans("Expense::expense.alert.delete_error"), 'Error');
            return redirect()->route('expense.list');
        }
    }

    public function removeCached()
    {
        Cache::forget('expense');
    }


}
