<?php
/*
| -------------------------------------------------------------
| AUTHOR:           INILABS TEAM
| -------------------------------------------------------------
| EMAIL:            info@inilabs.net or labs.ini@gmail.com
| -------------------------------------------------------------
| WEBSITE:          http://inilabs.net
| -------------------------------------------------------------
*/
namespace Modules\ExportImport\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Artisan;
use Modules\ExportImport\Requests\ExportImportRequest;
use Alert, Cache;
use Exception;
use Log;
use Storage;

class ExportImportController extends Controller
{

    public function index()
    {
        if (!count(config('laravel-backup.backup.destination.disks'))) {
            Alert::error(trans("ExportImport::exportimport.alert.configuration_error"), 'error');
            return redirect()->route('admin.home');
        }

        $this->data['backups'] = [];

        foreach (config('laravel-backup.backup.destination.disks') as $disk_name) {
            $disk = Storage::disk($disk_name);
            $adapter = $disk->getDriver()->getAdapter();
            $files = $disk->allFiles();

            // make an array of backup files, with their filesize and creation date
            foreach ($files as $k => $f) {
                // only take the zip files into account
                if (substr($f, -4) == '.zip' && $disk->exists($f)) {
                    $this->data['backups'][] = [
                        'file_path'     => $f,
                        'file_name'     => str_replace('backups/', '', $f),
                        'file_size'     => $disk->size($f),
                        'last_modified' => $disk->lastModified($f),
                        'disk'          => $disk_name,
                        'download'      => ($adapter instanceof \League\Flysystem\Adapter\Local) ? true : false,
                    ];
                }
            }
        }

        // reverse the backups, so the newest one would be on top
        $this->data['backups'] = array_reverse($this->data['backups']);
        $this->data['title'] = 'Backups';
        return view("ExportImport::exportimportCreate", $this->data);
    }

    public function create(ExportImportRequest $request)
    {
        if(config('core.demosite')) {
            Alert::warning('Disabled', trans('Core::demo.message'));
            return redirect()->route('export.create');
        }

        try {
            ini_set('max_execution_time', 300);
            // start the backup process
            if ($request->exportFormat=='db') {
                Artisan::call('backup:run',
                    [
                        '--only-db' => '1',
                    ]
                );
            } elseif ($request->exportFormat=='file') {
                Artisan::call('backup:run',
                    [
                        '--only-files' => '1',
                    ]
                );
            } elseif ($request->exportFormat=='full') {
                Artisan::call('backup:run');
            }

            $output = Artisan::output();
            // return the results as a response to the ajax call
        } catch (Exception $e) {
            Response::make($e->getMessage(), 500);
        }
        Alert::success(trans("ExportImport::exportimport.alert.create_success"), 'success');
        return redirect()->route('export.create');
    }

    /**
     * Downloads a backup zip file.
     */
    public function download()
    {
        if(config('core.demosite')) {
            Alert::warning('Disabled', trans('Core::demo.message'));
            return redirect()->route('export.create');
        }

        $disk = Storage::disk(\Request::input('disk'));
        $file_name = \Request::input('file_name');
        $adapter = $disk->getDriver()->getAdapter();

        if ($adapter instanceof \League\Flysystem\Adapter\Local) {
            $storage_path = $disk->getDriver()->getAdapter()->getPathPrefix();

            if ($disk->exists($file_name)) {
                return response()->download($storage_path.$file_name);
            } else {
                abort(404);
            }
        } else {
            abort(404);
        }
    }

    /**
     * Deletes a backup file.
     */
    public function delete()
    {
        if(config('core.demosite')) {
            Alert::warning('Disabled', trans('Core::demo.message'));
            return redirect()->route('export.create');
        }

        $disk = Storage::disk(\Request::input('disk'));
        $file_name = \Request::input('file_name');
        if ($disk->exists($file_name)) {
            $disk->delete($file_name);

            Alert::success('Success', trans("ExportImport::exportimport.alert.delete_success"))->autoclose(2600);
            return redirect()->route('export.create');
        } else {
            abort(404);
        }
    }

}
