<?php
/*
| -------------------------------------------------------------
| iBill - Simplest and Best Billing & Invoice Manager
| -------------------------------------------------------------
| AUTHOR:           INILABS TEAM
| -------------------------------------------------------------
| EMAIL:            info@inilabs.net
| -------------------------------------------------------------
| COPYRIGHT:        RESERVED BY INILABS IT
| -------------------------------------------------------------
| WEBSITE:          http://inilabs.net
| -------------------------------------------------------------
*/
namespace Modules\Iauth\Controllers;

use Carbon\Carbon;
use ConsoleTVs\Charts\Facades\Charts;
use Illuminate\Validation\Rules\In;
use Modules\Bank\Models\Bank;
use Modules\Expense\Models\Expense;
use Modules\Iauth\Events\Access\User\History\UserDeleted;
use App\Http\Controllers\Controller;
use Modules\Iauth\Repositories\Access\Role\RoleRepositoryContract;
use Modules\Iauth\Repositories\Access\User\UserRepositoryContract;
use Modules\Iauth\Requests\UpdateUserPasswordRequest;
use Modules\Iauth\Requests\UserCreateAndUpdateRequest;
use Modules\Iauth\Requests\AuthRequest;

use Illuminate\Http\Request;
use Modules\Iauth\Models\Access\User\User;
use Modules\Iauth\Models\Access\Role\Role;
use Modules\Invoice\Models\Invoice;
use Modules\Invoice\Models\Payment;
use Modules\Notice\Models\Notice;
use Modules\Post\Models\Post;
use Modules\Todo\Repositories\TodoRepository;
use Yajra\Datatables\Facades\Datatables;
use Cache, Auth, Alert, Image, File;

class IauthController extends Controller
{

	protected $redirectPath = '/admin/';


	protected $loginPath = '/admin/login';

    protected $userRepo;

    protected $roleRepo;

    public function __construct(UserRepositoryContract $user, RoleRepositoryContract $role)
    {
        $this->userRepo     = $user;
        $this->roleRepo     = $role;

        if(config('core.havefrontend')) {
            $this->loginPath = '/';
        }
    }

	public function index()
	{
		return view("backend::pages.registration");
	}

    public function home()
    {
        $user = auth()->user();

        $allInvoices = Cache::remember('all_invoices_'.$user->id,60, function () use ($user) {
            if (auth()->user()->hasRole('Admin')) {
                return Invoice::with('client')->orderBy('created_at', 'desc')->get();
            } else {
                return Invoice::where('client_id', $user->id)->with('client')->orderBy('created_at', 'desc')->get();
            }
        });

        $payments = Cache::remember('payments',60, function () use ($user){
            if (auth()->user()->hasRole('Admin')) {
                return Payment::with('user', 'invoice')->orderBy('created_at', 'desc')->get();
            } else {
                return Payment::where('user_id', $user->id)->with('user', 'invoice')->orderBy('created_at', 'desc')->get();
            }
        });

        $invoices = Cache::remember('invoice', 30, function () {
            return Invoice::selectRaw('month(create_date) as month, sum(grand_total) as sum')
                ->groupBy('month')
                ->orderBy('month')
                ->get()->pluck('sum')->toArray();
        });

        $expenses = Cache::remember('expense', 30, function () {
            return Expense::selectRaw('month(purchased_date) as month, sum(purchased_amount) as sum')
                ->groupBy('month')
                ->orderBy('month')
                ->get()->pluck('sum')->toArray();
        });

        $bankBalance = Cache::remember('bankBalance', 30, function (){
            return Bank::with(['bankPayments.payment', 'deposits'])->get();
        });

        $expense = Expense::whereYear('created_at', date('Y'))->get()->toArray();
        $income = Payment::whereYear('created_at', date('Y'))->get()->toArray();
        $months = array(
            1 => 'January',
            'February',
            'March',
            'April',
            'May',
            'June',
            'July ',
            'August',
            'September',
            'October',
            'November',
            'December',
        );

        $expenseMonthAndDay = array();
        $expenseMonthTotal  = array();
        foreach ($expense as $key => $value) {
             $expensemonth = Carbon::parse($value['created_at'])->format('m');
             $expenseday = Carbon::parse($value['created_at'])->format('d');
             if(!isset($expenseMonthAndDay[(int)$expensemonth][$expenseday])) {
                 $expenseMonthAndDay[(int)$expensemonth][$expenseday] = 0;
             }

            $expenseMonthAndDay[(int)$expensemonth][$expenseday] += $value['purchased_amount'];

            if(!isset($expenseMonthTotal[(int)$expensemonth])) {
                $expenseMonthTotal[(int)$expensemonth] = 0;
            }
            $expenseMonthTotal[(int)$expensemonth] += $value['purchased_amount'];
        }

        $incomeMonthAndDay = array();
        $incomeMonthTotal  = array();
        foreach ($income as $key => $value) {
            $paymentmonth = Carbon::parse($value['created_at'])->format('m');
            $paymentday = Carbon::parse($value['created_at'])->format('d');

            if(!isset($incomeMonthAndDay[(int)$paymentmonth][$paymentday])) {
                $incomeMonthAndDay[(int)$paymentmonth][$paymentday] = 0;
            }

            $incomeMonthAndDay[(int)$paymentmonth][$paymentday] += $value['paid_amount'];

            if(!isset($incomeMonthTotal[(int)$paymentmonth])) {
                $incomeMonthTotal[(int)$paymentmonth] = 0;
            }
            $incomeMonthTotal[(int)$paymentmonth] += $value['paid_amount'];
        }

        if (auth()->user()->hasRole('Client')) {
            $relationTableData = $user->hasOneRelation('Modules\Client\Models\Client')->first();
            if ($relationTableData == null) {
                $relationTableData = Client::create([
                    'company' => "",
                    'user_id' => $user->id
                ]);
            }
            return view('backend::pages.dashboard')
                ->withInvoices($allInvoices)
                ->withExpenses($expenses)
                ->withPayments($payments)
                ->withUser($user)
                ->withClient($relationTableData);
        }

        return view('backend::pages.dashboard', compact('incomeMonthAndDay', 'months', 'incomeMonthTotal', 'expenseMonthAndDay', 'expenseMonthTotal', 'bankBalance'))
            ->withInvoices($allInvoices)
            ->withExpenses($expenses)
            ->withPayments($payments)
            ->withUser($user);

    }

    public function dayWiseExpenseOrIncome(Request $request)
    {
        $type = $request->type;
        $monthID = $request->monthID;
        $days = cal_days_in_month(CAL_GREGORIAN, $monthID, date('Y'));
        $dayWiseData = json_decode($request->dayWiseData, true);

        $showChartData = array();
        for ($i=1; $i <= $days; $i++) {
            if(!isset($dayWiseData[$i])) {
                $showChartData[$i] = 0;
            } else {
                $showChartData[$i] = $dayWiseData[$i];
            }
        }

        echo json_encode($showChartData);
    }

	public function login(AuthRequest $request)
	{
	    $result = $this->userRepo->loginUserNameOrEmail($request);

        if($result == 1) {
            Alert::success('Success', trans("Iauth::login.login_success"));
            return redirect()->intended($this->redirectPath);
        } elseif ($result == -1) {
            flash(trans("Iauth::login.account_blocked"), 'warning');
            return redirect(route('admin.login'));
        }
        $haveUserEmailOrUserName = User::where('email', '=', $request['emailorusername'])->orWhere('username', '=', $request['emailorusername'])->first();
		return redirect(route('admin.login'))
					->withInput($request->only('emailorusername'))
					->withErrors([
						'emailorusername' => $haveUserEmailOrUserName == null ? trans("Iauth::login.login_error") : '',
						'password' => trans("Iauth::login.login_error"),
					]);
	}

	public function logout()
	{
        Cache::flush();
		Auth::logout();
        session()->forget('notice-button-click');
        flash(trans("Iauth::login.logout_success"), 'success');
		return redirect($this->loginPath);
	}

	public function userList()
	{
        return view('Iauth::user.userList');
	}

	public function userDataTable()
	{
        $allRoles       = Role::all()->pluck('name')->toArray();
        $separatedRole  = config("core.roles.separated");
        $roles          = array_diff($allRoles, $separatedRole);

        return DataTables::of($this->userRepo->getAllUsersByRole($roles))
            ->addColumn('roles', function($user) {
                $roles = [];

                if ($user->roles()->count() > 0) {
                    foreach ($user->roles as $role) {
                        array_push($roles, $role->name);
                    }

                    return implode("<br/>", $roles);
                } else {
                    return 'None';
                }
            })
            ->addColumn('photo', function ($user) {
                return "<img class=\"img-circle\" src=\"".url($user->img)."\" alt=\"$user->name Img\" width=\"42\" hight=\"42\">";
            })
            ->addColumn('name', function($user) {
                return $user->name;
            })
            ->addColumn('actions', function($user) {
                return $user->action_buttons;
            })
            ->make(true);
	}

	public function create(UserCreateAndUpdateRequest $request, $redirectRoute = 'user.list')
	{
        $result = $this->userRepo->create($request->except('assignees_roles'), $request->only('assignees_roles'));

        if($result) {
            $this->removeCached();
            Alert::success(trans("Iauth::user.alert.create_success"), 'success');
            return redirect(route($redirectRoute));
        }
	}

    public function add()
    {
        return view('Iauth::user.userCreate')
                ->withRoles($this->roleRepo->getAllRoles());
	}

	public function edit(User $user)
    {
        return view('Iauth::user.userEdit')
                ->withUser($user)
                ->withUserRoles($user->roles->pluck('id')->all())
                ->withRoles($this->roleRepo->getAllRoles());
	}

	public function view($user)
    {
        if(is_numeric($user)) {
            $user = User::find($user);

            return view('Iauth::user.userView')
                    ->withUser($user)
                    ->withRoles($this->roleRepo->getAllRoles());
        } else {
            $user = User::where('username', '=', $user)->first();

            if($user == null) {
                return abort(404);
            }

            return view('Iauth::user.userView')
                ->withUser($user)
                ->withRoles($this->roleRepo->getAllRoles());
        }
	}

	public function update(User $user, UserCreateAndUpdateRequest $request)
	{
        if ($request->hasFile('image')) {
            if(!is_dir(base_path(config('core.users.profile_image_path')))) {
                File::makeDirectory(base_path(config('core.users.profile_image_path')));
            }
            Image::make($request->file('image'))->resize(462, 462)->save(config('core.users.profile_image_path').$user->username.'.jpg');
            $request['img'] = config('core.users.profile_image_path').$user->username.'.jpg';
        }
        $this->userRepo->update($user,
            $request->except('assignees_roles', 'image'),
            $request->only('assignees_roles')
        );
        \Artisan::call('optimize');
        \Artisan::call('cache:clear');
        \Artisan::call('view:clear');
        $this->removeCached();
        Alert::success(trans("Iauth::user.alert.update_success"), 'success');
        return redirect(route('user.list'));
	}

    /**
     * @param User $user
     * @param $status
     * @param ManageUserRequest $request
     * @return mixed
     */
    public function status(User $user, $status)
    {
        $this->userRepo->changeActiveStatus($user, $status);
        $this->removeCached();
        Alert::success(trans("Iauth::user.alert.update_success"), 'success');
        return redirect()->back();
    }

    /**
     * @param User $user
     * @param ManageUserRequest $request
     * @return mixed
     */
    public function changePassword(User $user)
    {
        return view('Iauth::user.changePassword')
            ->withUser($user);
    }

    /**
     * @param User $user
     * @param UpdateUserPasswordRequest $request
     * @return mixed
     */
    public function updatePassword(User $user, UpdateUserPasswordRequest $request)
    {
        if(config('core.demosite')) {
            Alert::warning('Disabled', trans('Core::demo.message'));
            return redirect()->back();
        }

        $this->userRepo->updatePassword($user, $request->all());
        $this->removeCached();
        Alert::success(trans("Iauth::user.alert.change_pass_success"), 'success');
        return redirect()->route('user.list');
    }

    public function delete($user)
    {
        $user = User::findOrFail($user);
        $user->delete();
        $this->removeCached();
        event(new UserDeleted($user));
        Alert::success(trans("Iauth::user.alert.delete_success"), 'success');
        return redirect()->back();
	}

    public function multiDelete(Request $request)
    {
        $ids = explode(',', $request['ids']);
        foreach ($ids as $id) {
            if($id == auth()->id() && auth()->user()->hasPermission('admin@user-status')) continue;
            $user = User::findOrFail($id);
            $user->delete();
            event(new UserDeleted($user));
        }
        $this->removeCached();

        Alert::success(trans("Iauth::user.alert.delete_success"), 'success');
        return redirect()->back();
	}

    public function profile()
    {
        $user               = User::find(auth()->user()->id);
        if (auth()->user()->hasRole('Client')) {
            $relationTableData = $user->hasOneRelation('Modules\Client\Models\Client')->first();
            if ($relationTableData == null) {
                $relationTableData = Client::create([
                    'company' => "",
                    'user_id' => $user->id
                ]);
            }
            $invoices = Invoice::where('client_id', $user->id)->with('client')->orderBy('created_at', 'desc')->get();
            return $user != null ? view('Client::clientProfileView')
                ->withUser($user)
                ->withClient($relationTableData)
                ->withInvoices($invoices) :  abort(404);
        } else {
            return view('Iauth::user.userProfile')
                ->withUser($user)
                ->withRoles($this->roleRepo->getAllRoles());
        }
	}

    public function profileEdit($user)
    {
        if(is_numeric($user)) {
            $user = User::find($user);

            if($user->id != auth()->user()->id)
                abort(404);

            if($this->loadClientView($user)) {
                $all_countries = array("Afghanistan", "Albania", "Algeria", "American Samoa", "Andorra", "Angola", "Anguilla", "Antarctica", "Antigua and Barbuda", "Argentina", "Armenia", "Aruba", "Australia", "Austria", "Azerbaijan", "Bahamas", "Bahrain", "Bangladesh", "Barbados", "Belarus", "Belgium", "Belize", "Benin", "Bermuda", "Bhutan", "Bolivia", "Bosnia and Herzegowina", "Botswana", "Bouvet Island", "Brazil", "British Indian Ocean Territory", "Brunei Darussalam", "Bulgaria", "Burkina Faso", "Burundi", "Cambodia", "Cameroon", "Canada", "Cape Verde", "Cayman Islands", "Central African Republic", "Chad", "Chile", "China", "Christmas Island", "Cocos (Keeling) Islands", "Colombia", "Comoros", "Congo", "Congo, the Democratic Republic of the", "Cook Islands", "Costa Rica", "Cote d'Ivoire", "Croatia (Hrvatska)", "Cuba", "Cyprus", "Czech Republic", "Denmark", "Djibouti", "Dominica", "Dominican Republic", "East Timor", "Ecuador", "Egypt", "El Salvador", "Equatorial Guinea", "Eritrea", "Estonia", "Ethiopia", "Falkland Islands (Malvinas)", "Faroe Islands", "Fiji", "Finland", "France", "France Metropolitan", "French Guiana", "French Polynesia", "French Southern Territories", "Gabon", "Gambia", "Georgia", "Germany", "Ghana", "Gibraltar", "Greece", "Greenland", "Grenada", "Guadeloupe", "Guam", "Guatemala", "Guinea", "Guinea-Bissau", "Guyana", "Haiti", "Heard and Mc Donald Islands", "Holy See (Vatican City State)", "Honduras", "Hong Kong", "Hungary", "Iceland", "India", "Indonesia", "Iran (Islamic Republic of)", "Iraq", "Ireland", "Israel", "Italy", "Jamaica", "Japan", "Jordan", "Kazakhstan", "Kenya", "Kiribati", "Korea, Democratic People's Republic of", "Korea, Republic of", "Kuwait", "Kyrgyzstan", "Lao, People's Democratic Republic", "Latvia", "Lebanon", "Lesotho", "Liberia", "Libyan Arab Jamahiriya", "Liechtenstein", "Lithuania", "Luxembourg", "Macau", "Macedonia, The Former Yugoslav Republic of", "Madagascar", "Malawi", "Malaysia", "Maldives", "Mali", "Malta", "Marshall Islands", "Martinique", "Mauritania", "Mauritius", "Mayotte", "Mexico", "Micronesia, Federated States of", "Moldova, Republic of", "Monaco", "Mongolia", "Montserrat", "Morocco", "Mozambique", "Myanmar", "Namibia", "Nauru", "Nepal", "Netherlands", "Netherlands Antilles", "New Caledonia", "New Zealand", "Nicaragua", "Niger", "Nigeria", "Niue", "Norfolk Island", "Northern Mariana Islands", "Norway", "Oman", "Pakistan", "Palau", "Panama", "Papua New Guinea", "Paraguay", "Peru", "Philippines", "Pitcairn", "Poland", "Portugal", "Puerto Rico", "Qatar", "Reunion", "Romania", "Russian Federation", "Rwanda", "Saint Kitts and Nevis", "Saint Lucia", "Saint Vincent and the Grenadines", "Samoa", "San Marino", "Sao Tome and Principe", "Saudi Arabia", "Senegal", "Seychelles", "Sierra Leone", "Singapore", "Slovakia (Slovak Republic)", "Slovenia", "Solomon Islands", "Somalia", "South Africa", "South Georgia and the South Sandwich Islands", "Spain", "Sri Lanka", "St. Helena", "St. Pierre and Miquelon", "Sudan", "Suriname", "Svalbard and Jan Mayen Islands", "Swaziland", "Sweden", "Switzerland", "Syrian Arab Republic", "Taiwan, Province of China", "Tajikistan", "Tanzania, United Republic of", "Thailand", "Togo", "Tokelau", "Tonga", "Trinidad and Tobago", "Tunisia", "Turkey", "Turkmenistan", "Turks and Caicos Islands", "Tuvalu", "Uganda", "Ukraine", "United Arab Emirates", "United Kingdom", "United States", "United States Minor Outlying Islands", "Uruguay", "Uzbekistan", "Vanuatu", "Venezuela", "Vietnam", "Virgin Islands (British)", "Virgin Islands (U.S.)", "Wallis and Futuna Islands", "Western Sahara", "Yemen", "Yugoslavia", "Zambia", "Zimbabwe");
                $countries = array_combine($all_countries, $all_countries);
                return view('Client::clientProfileEdit', compact('user', 'countries'));
            } else {
                return view('Iauth::user.userProfileEdit')
                    ->withUser($user);
            }
        } else {
            $user = User::where('username', '=', $user)->first();

            if($user == null) {
                return abort(404);
            }

            if($user->id != auth()->user()->id)
                abort(404);

            if($this->loadClientView($user)) {
                $all_countries = array("Afghanistan", "Albania", "Algeria", "American Samoa", "Andorra", "Angola", "Anguilla", "Antarctica", "Antigua and Barbuda", "Argentina", "Armenia", "Aruba", "Australia", "Austria", "Azerbaijan", "Bahamas", "Bahrain", "Bangladesh", "Barbados", "Belarus", "Belgium", "Belize", "Benin", "Bermuda", "Bhutan", "Bolivia", "Bosnia and Herzegowina", "Botswana", "Bouvet Island", "Brazil", "British Indian Ocean Territory", "Brunei Darussalam", "Bulgaria", "Burkina Faso", "Burundi", "Cambodia", "Cameroon", "Canada", "Cape Verde", "Cayman Islands", "Central African Republic", "Chad", "Chile", "China", "Christmas Island", "Cocos (Keeling) Islands", "Colombia", "Comoros", "Congo", "Congo, the Democratic Republic of the", "Cook Islands", "Costa Rica", "Cote d'Ivoire", "Croatia (Hrvatska)", "Cuba", "Cyprus", "Czech Republic", "Denmark", "Djibouti", "Dominica", "Dominican Republic", "East Timor", "Ecuador", "Egypt", "El Salvador", "Equatorial Guinea", "Eritrea", "Estonia", "Ethiopia", "Falkland Islands (Malvinas)", "Faroe Islands", "Fiji", "Finland", "France", "France Metropolitan", "French Guiana", "French Polynesia", "French Southern Territories", "Gabon", "Gambia", "Georgia", "Germany", "Ghana", "Gibraltar", "Greece", "Greenland", "Grenada", "Guadeloupe", "Guam", "Guatemala", "Guinea", "Guinea-Bissau", "Guyana", "Haiti", "Heard and Mc Donald Islands", "Holy See (Vatican City State)", "Honduras", "Hong Kong", "Hungary", "Iceland", "India", "Indonesia", "Iran (Islamic Republic of)", "Iraq", "Ireland", "Israel", "Italy", "Jamaica", "Japan", "Jordan", "Kazakhstan", "Kenya", "Kiribati", "Korea, Democratic People's Republic of", "Korea, Republic of", "Kuwait", "Kyrgyzstan", "Lao, People's Democratic Republic", "Latvia", "Lebanon", "Lesotho", "Liberia", "Libyan Arab Jamahiriya", "Liechtenstein", "Lithuania", "Luxembourg", "Macau", "Macedonia, The Former Yugoslav Republic of", "Madagascar", "Malawi", "Malaysia", "Maldives", "Mali", "Malta", "Marshall Islands", "Martinique", "Mauritania", "Mauritius", "Mayotte", "Mexico", "Micronesia, Federated States of", "Moldova, Republic of", "Monaco", "Mongolia", "Montserrat", "Morocco", "Mozambique", "Myanmar", "Namibia", "Nauru", "Nepal", "Netherlands", "Netherlands Antilles", "New Caledonia", "New Zealand", "Nicaragua", "Niger", "Nigeria", "Niue", "Norfolk Island", "Northern Mariana Islands", "Norway", "Oman", "Pakistan", "Palau", "Panama", "Papua New Guinea", "Paraguay", "Peru", "Philippines", "Pitcairn", "Poland", "Portugal", "Puerto Rico", "Qatar", "Reunion", "Romania", "Russian Federation", "Rwanda", "Saint Kitts and Nevis", "Saint Lucia", "Saint Vincent and the Grenadines", "Samoa", "San Marino", "Sao Tome and Principe", "Saudi Arabia", "Senegal", "Seychelles", "Sierra Leone", "Singapore", "Slovakia (Slovak Republic)", "Slovenia", "Solomon Islands", "Somalia", "South Africa", "South Georgia and the South Sandwich Islands", "Spain", "Sri Lanka", "St. Helena", "St. Pierre and Miquelon", "Sudan", "Suriname", "Svalbard and Jan Mayen Islands", "Swaziland", "Sweden", "Switzerland", "Syrian Arab Republic", "Taiwan, Province of China", "Tajikistan", "Tanzania, United Republic of", "Thailand", "Togo", "Tokelau", "Tonga", "Trinidad and Tobago", "Tunisia", "Turkey", "Turkmenistan", "Turks and Caicos Islands", "Tuvalu", "Uganda", "Ukraine", "United Arab Emirates", "United Kingdom", "United States", "United States Minor Outlying Islands", "Uruguay", "Uzbekistan", "Vanuatu", "Venezuela", "Vietnam", "Virgin Islands (British)", "Virgin Islands (U.S.)", "Wallis and Futuna Islands", "Western Sahara", "Yemen", "Yugoslavia", "Zambia", "Zimbabwe");
                $countries = array_combine($all_countries, $all_countries);
                return view('Client::clientProfileEdit', compact('user', 'countries'));
            } else {
                return view('Iauth::user.userProfileEdit')
                    ->withUser($user);
            }
        }
    }

    public function loadClientView(&$user)
    {
        $separatedRoles = config('core.roles.separated');
        $collection = $user->roles->pluck('name')->toArray();
        $result = array_intersect($collection, $separatedRoles);

        if (count($result) == 1) {
            $relationTableData = $user->hasOneRelation('Modules\Client\Models\Client')->first();
            $inputFieldPairToModelRelation = [
                'company' => 'company',
                'phone' => 'phone',
                'address_line_one' => 'address_line_one',
                'address_line_two' => 'address_line_two',
                'city' => 'city',
                'zip_code' => 'zip_code',
                'state' => 'state',
                'country' => 'country',
                'client_id' => 'id'
            ];

            foreach ($inputFieldPairToModelRelation as $inputViewIndex => $relationIndex) {
                $user[$inputViewIndex] = $relationTableData[$relationIndex];
            }
            return true;
        } else {
            return false;
        }
    }

    public function updateProfileEdit(User $user, UserCreateAndUpdateRequest $request)
    {
        if($user->id != auth()->user()->id)
            abort(404);

        if ($request->hasFile('image')) {
            if(!is_dir(base_path(config('core.users.profile_image_path')))) {
                File::makeDirectory(base_path(config('core.users.profile_image_path')));
            }
            Image::make($request->file('image'))->resize(462, 462)->save(config('core.users.profile_image_path').$user->username.'.jpg');
            $request['img'] = config('core.users.profile_image_path').$user->username.'.jpg';
        }
        $this->userRepo->update($user,
            $request->except('assignees_roles', 'image'),
            $request->only('assignees_roles')
        );
        $this->removeCached();
        Alert::success(trans("Iauth::user.alert.update_success"), 'success');
        return redirect(route('profile'));
    }

    public function changeProfilePassword()
    {
        $user = User::find(auth()->user()->id);
        return view('Iauth::user.changeProfilePassword')
                    ->withUser($user);
    }

    public function updateProfilePassword(UpdateUserPasswordRequest $request)
    {
        if(config('core.demosite')) {
            Alert::warning('Disabled', trans('Core::demo.message'));
            return redirect()->back();
        }
        $user = User::find(auth()->user()->id);
        $this->userRepo->updatePassword($user, $request->all());
        $this->removeCached();
        Alert::success(trans("Iauth::user.alert.change_pass_success"), 'success');
        return redirect()->route('profile');
    }

    public function removeCached()
    {
        Cache::forget('only_trashed_users');
        Cache::forget('active_user');
        Cache::forget('all_users');
        Cache::forget('get_user_by_role');
        Cache::forget('users');
    }

}
