<?php
/*
| -------------------------------------------------------------
| iBill - Simplest and Best Billing & Invoice Manager
| -------------------------------------------------------------
| AUTHOR:           INILABS TEAM
| -------------------------------------------------------------
| EMAIL:            info@inilabs.net
| -------------------------------------------------------------
| COPYRIGHT:        RESERVED BY INILABS IT
| -------------------------------------------------------------
| WEBSITE:          http://inilabs.net
| -------------------------------------------------------------
*/
namespace Modules\Iauth\Repositories\Access\User;

use Modules\Iauth\Events\Access\User\Mail\UserRegistered;
use Modules\Iauth\Events\Access\User\History\UserCreated;
use Modules\Iauth\Events\Access\User\History\UserUpdated;
use Modules\Iauth\Events\Access\User\History\UserPasswordChanged;
use Modules\Iauth\Models\Access\User\User;
use Modules\Iauth\Models\Access\Role\Role;
use Illuminate\Support\Facades\DB;
use App\Exceptions\GeneralException;
use Auth, Image, Cache;
/**
 * Class EloquentUserRepository
 * @package App\Repositories\User
 */
class EloquentUserRepository implements UserRepositoryContract
{
    /**
     * @param int $status
     * @param bool $trashed
     * @return mixed
     */
    public function getAllUsers($status = false, $trashed = false)
    {
        /**
         * Note: You must return deleted_at or the User getActionButtonsAttribute won't
         * be able to differentiate what buttons to show for each row.
         */
        if ($trashed == "true") {
            return Cache::remember('only_trashed_users', 60, function () {
                return User::onlyTrashed()
                    ->all()
                    ->get();
            });
        } elseif($status == "true") {
            return Cache::remember('active_user', 60, function () {
                return User::all()
                    ->where('status', 1);
            });
        }

        return Cache::remember('all_users', 60, function () {
            return User::all();
        });
    }

    /**
     * @param $role
     * @param $operator
     * @param $column
     * return collection of data
     */
    public function getAllUsersByRole( $role, $trashed = false, $column = 'name', $operator = '=')
    {
        if(is_array($role) && is_numeric($role[0])) {
            $column = 'id';
        } elseif (is_numeric($role)) {
            $column = 'id';
        }

        if ($trashed) {
            return User::whereHas(config('core.acl.roles_table'), function ($query) use ($column, $operator, $role) {
                if (is_array($role)) {
                    $query->whereIn(config('core.acl.roles_table').'.'.$column, $role);
                } else {
                    $query->where(config('core.acl.roles_table').'.'.$column, $operator, $role);
                }
            })->orHas(config('core.acl.roles_table'), '=', 0)->get();
        } else {
            return User::whereHas(config('core.acl.roles_table'), function ($query) use ($column, $operator, $role) {
                if (is_array($role)) {
                    $query->whereIn(config('core.acl.roles_table').'.'.$column, $role);
                } else {
                    $query->where(config('core.acl.roles_table').'.'.$column, $operator, $role);
                }
            })->orHas(config('core.acl.roles_table'), '=', 0)->where('deleted_at', '=', null)->get();
        }
    }

    /**
     * @param $input
     * @param $roles
     * @return mixed
     */
    public function create($input, $roles = [], $model = null)
    {
        $user = $this->createUserStub($input);

		return DB::transaction(function() use ($user, $roles, $input, $model) {
			if ($user->save()) {
                if($model != null) {
                    $model = new $model;
                    $input['user_id'] = $user->id;
                    $model->fill($input)->save();
                }
                if (!empty($input['image'])) {
                    if(!is_dir(base_path(config('core.users.profile_image_path')))) {
                        File::makeDirectory(base_path(config('core.users.profile_image_path')));
                    }
                    Image::make($input['image'])->resize(462, 462)->save(config('core.users.profile_image_path').$user->username.'.jpg');
                }

			    if(isset($roles['assignees_roles']) && count($roles['assignees_roles']) > 0) {
                    $user->attachRoles($roles['assignees_roles']);
                }
                if (env('MAIL_USERNAME')!= null) {
                    event(new UserRegistered($user));
                }
                if (auth()->user()) {
                    event(new UserCreated($user));
                }
				return true;
			}

		});
    }

    /**
     * @param  $input
     * @return 0 = not found any data for login, -1 = found data but it's blocked, 1 = login
     */
    public function loginUserNameOrEmail($input)
    {
        $remember = isset($input['remember']) && $input['remember'] == 'on' ? true : false;

        $emailCheck = [
            'email'     => $input['emailorusername'],
            'password'  => $input['password'],
        ];

        $usernameCheck = [
            'username'  => $input['emailorusername'],
            'password'  => $input['password'],
        ];

        if( Auth::attempt($emailCheck) || Auth::attempt($usernameCheck) ) {
            if(Auth::user()->status == 0) {
                Auth::logout();
                return -1;
            }

            Auth::loginUsingId(Auth::user()->id, $remember);
            return 1;
        }

        return 0;
    }

    /**
     * @param User $user
     * @param $input
     * @param $roles
     * @return bool
     * @throws GeneralException
     */
    public function update(User $user, $input, $roles, $model = null)
    {
        $input['status']            = isset($input['status']) && $input['status'] == '1' ? 1 : 0;
        $input['is_valid_email']    = isset($input['is_valid_email']) && $input['is_valid_email'] == '1' ? 1 : 0;
        if($model != null) {
            $model->fill($input)->save();
            foreach ($model->getfillable() as $key) {
                unset($input[$key]);
            }
        }

        $user->fill($input)->save();

        if(isset($roles['assignees_roles']) && (count($roles['assignees_roles']) > 0 )) {
            $this->flushRoles($roles, $user);
        }
        event(new UserUpdated($user));
        return true;
    }

    /**
     * @param $roles
     * @param $user
     */
    private function flushRoles($roles, $user)
    {
        $user->detachRoles($user->roles);
        $user->attachRoles($roles['assignees_roles']);
    }

    /**
     * @param  User $user
     * @param  $status
     * @throws GeneralException
     * @return bool
     */
    public function changeActiveStatus(User $user, $status)
    {
        if (auth()->id() == $user->id && $status == 0) {
            return 0;
        }

        $user->status = $status;

        if ($user->save()) {
            return true;
        }

    }

    /**
     * @param  User $user
     * @param  $input
     * @return bool
     */
    public function updatePassword(User $user, $input)
    {
        $user->password = bcrypt($input['password']);

        if ($user->save()) {
            event(new UserPasswordChanged($user));
            return true;
        }

        return false;
    }

    /**
     * @param  $input
     * @return mixed
     */
    private function createUserStub($input)
    {
        $user                    = new User;
        $user->username          = $input['username'];
        $user->first_name        = isset($input['first_name']) ? $input['first_name'] : null;
        $user->last_name         = isset($input['last_name']) ? $input['last_name'] : null;
        $user->img               = !empty($input['image']) ? config('core.users.profile_image_path').$input['username'].'.jpg' : config('core.users.profile_image_path').'default.png';
        $user->email             = $input['email'];
        $user->password          = bcrypt($input['password']);
        $user->status            = isset($input['status']) ? 1 : config('core.users.active_user_status');
        $user->confirmation_code = md5(uniqid(mt_rand(), true));
        $user->is_valid_email    = isset($input['is_valid_email']) ? 1 : 0;
        return $user;
    }

}
