<?php
/*
| -------------------------------------------------------------
| AUTHOR:           INILABS TEAM
| -------------------------------------------------------------
| EMAIL:            info@inilabs.net or labs.ini@gmail.com
| -------------------------------------------------------------
| WEBSITE:          http://inilabs.net
| -------------------------------------------------------------
*/

namespace Modules\Product\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Modules\Product\Models\Product;
use Modules\Product\Repositories\ProductRepository;
use Modules\Product\Requests\ProductRequest;
use Yajra\Datatables\Facades\Datatables;
use Alert, Cache;

class ProductController extends Controller
{
    protected $productRepo;

    function __construct(ProductRepository $product)
    {
        $this->productRepo = $product;
    }

    public function index()
    {
        return view("Product::productList");
    }

    public function productDataTable()
    {
        return DataTables::of($this->productRepo->findAll(['*']))
            ->editColumn('note', function ($product){
                $lenth = strlen($product->note);
                if ($lenth > 99)
                    return substr($product->note, 0, 100)."...";
                else
                    return $product->note;
            })
            ->editColumn('amount', function ($product){
                return isite()->siteCurrencySymbol().number_format($product->amount, 2);
            })
            ->addColumn('actions', function ($product){
                return $product->action_buttons;
            })
            ->make(true);
    }

    public function add()
    {
        return view("Product::productCreate");
    }

    public function create(ProductRequest $request)
    {
        $input = $request->all();
        $input['user_id'] = auth()->user()->id;
        $result = $this->productRepo->create($input);

        if ($result) {
            $this->removeCached();
            Alert::success(trans("Product::product.alert.create_success"), 'success');
            return redirect()->route('product.list');
        }
    }

    public function edit(Product $product)
    {
        return view("Product::productEdit")->withProduct($product);
    }

    public function update(Product $product, ProductRequest $request)
    {
        $result = $this->productRepo->update($product, $request->all());

        if ($result) {
            $this->removeCached();
            Alert::success(trans("Product::product.alert.update_success"), 'success');
            return redirect()->route('product.list');
        }
    }

    public function status(Product $product)
    {
        $data = [];
        if($product->user_id == auth()->user()->id) {
            if ($product->status==1) {
                $data['status'] = 0;
            } else {
                $data['status'] = 1;
            }
            $this->productRepo->update($product->id, $data);
            $this->removeCached();
            Alert::success(trans("Product::product.alert.status_success"), 'success');
        } else {
            Alert::error(trans("Product::product.alert.status_error"), 'error');
        }

        return redirect()->route('product.list');
    }

    public function view(Product $product)
    {
        return $product != null ? view("Product::productView")->withProduct($product) :  abort(404);
    }

    public function delete(Product $product)
    {
        if($product->user_id == auth()->user()->id) {

            $this->productRepo->delete($product->id);

            $this->removeCached();
            Alert::success(trans("Product::product.alert.delete_success"), 'success');
            return redirect()->route('product.list');
        } else {
            $this->removeCached();
            Alert::error(trans("Product::product.alert.delete_error"), 'Error');
            return redirect()->route('product.list');
        }
    }

    public function removeCached()
    {
        Cache::forget('products');
    }


}
