<?php
/*
| -------------------------------------------------------------
| AUTHOR:           INILABS TEAM
| -------------------------------------------------------------
| EMAIL:            info@inilabs.net or labs.ini@gmail.com
| -------------------------------------------------------------
| WEBSITE:          http://inilabs.net
| -------------------------------------------------------------
*/
namespace Modules\Report\Controllers;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Modules\Expense\Models\Expense;
use Modules\Expense\Repositories\ExpenseRepository;
use Modules\Iauth\Models\Access\User\User;
use Modules\Iauth\Repositories\Access\User\UserRepositoryContract;
use Modules\Invoice\Models\Invoice;
use Modules\Invoice\Models\Payment;
use Modules\Invoice\Repositories\InvoiceRepository;
use Modules\Report\Requests\ReportRequest;
use Yajra\Datatables\Facades\Datatables;
use Alert, Cache;

class ReportController extends Controller
{
    protected $invoiceRepo;
    protected $userRepo;
    protected $expenseRepo;

    function __construct(InvoiceRepository $invoice,  UserRepositoryContract $client, ExpenseRepository $expense)
    {
        $this->invoiceRepo = $invoice;
        $this->userRepo = $client;
        $this->expenseRepo = $expense;
    }

    public function index()
    {
        $clients         = Cache::remember('all_clients',60, function () {
            return $this->userRepo->getAllUsersByRole('Client')->pluck('username', 'id')->toArray();
        });
        return view("Report::reportCreate")->withClients($clients);
    }

    public function payment()
    {
        $clients         = Cache::remember('all_clients',60, function () {
            return $this->userRepo->getAllUsersByRole('Client')->pluck('username', 'id')->toArray();
        });
        return view("Report::paymentReportCreate")->withClients($clients);
    }

    public function expense()
    {
        $users         = Cache::remember('all_users',60, function () {
            return User::all()->pluck('username', 'id')->toArray();
        });
        return view("Report::expenseReportCreate")->withUsers($users);
    }

    public function report(ReportRequest $request)
    {
        if($request) {
            return view("Report::reportList")->withRequest($request->all());
        } else {
            return redirect()->route('report.list');
        }
    }

    public function paymentReport(ReportRequest $request)
    {
        if($request) {
            $client = $request->client_id;
            $from   = $request->from_date;
            $to     = $request->to_date;

            if ($request->has('client_id')) {
                if ($from && $to) {
                    $payments = Payment::with('invoice')->where('user_id',$client)
                        ->whereBetween('created_at', array($from, $to))->get();
                } elseif ($from && $to==null) {
                    $payments = Payment::with('invoice')->where('user_id',$client)
                        ->whereBetween('created_at', array($from, date('Y-m-d')))->get();
                } elseif ($to && $from==null) {
                    $payments = Payment::with('invoice')->where('user_id',$client)
                        ->whereBetween('created_at', array($from, $to))->get();
                } else {
                    $payments = Payment::with('invoice')->where('user_id', $client)->get();
                }
            } else {
                if ($from && $to) {
                    $payments = Payment::with('invoice')->whereBetween('created_at', array($from, $to))->get();
                } elseif ($from && $to==null) {
                    $payments = Payment::with('invoice')->whereBetween('created_at', array($from, date('Y-m-d')))->get();
                } elseif ($to && $from==null) {
                    $payments = Payment::with('invoice')->whereBetween('created_at', array($from, $to))->get();
                } else {
                    $payments = Payment::with('invoice')->get();
                }
            }
            return view("Report::paymentReportList")->withPayments($payments)->withRequest($request->all());
        } else {
            return redirect()->route('payment.report.list');
        }
    }

    public function expenseReport(ReportRequest $request)
    {
        if($request) {
            return view("Report::expenseReportList")->withRequest($request->all());
        } else {
            return redirect()->route('expense.report.list');
        }
    }

    public function reportDataTable(ReportRequest $request)
    {
        if($request) {
            $client = $request->client_id;
            $from   = $request->from_date;
            $to     = $request->to_date;

            if ($request->has('client_id')) {
                if ($from && $to) {
                    $invoices = Invoice::with('payments')->where('client_id',$client)
                                ->whereBetween('created_at', array($from, $to))->get();
                } elseif ($from && $to==null) {
                    $invoices = Invoice::with('payments')->where('client_id',$client)
                        ->whereBetween('created_at', array($from, date('Y-m-d')))->get();
                } elseif ($to && $from==null) {
                    $invoices = Invoice::with('payments')->where('client_id',$client)
                        ->whereBetween('created_at', array($from, $to))->get();
                } else {
                    $invoices = $this->invoiceRepo->with(['payments'])->findWhere(['client_id', '=', $client]);
                }
            } else {
                if ($from && $to) {
                    $invoices = Invoice::with('payments')->whereBetween('created_at', array($from, $to))->get();
                } elseif ($from && $to==null) {
                    $invoices = Invoice::with('payments')->whereBetween('created_at', array($from, date('Y-m-d')))->get();
                } elseif ($to && $from==null) {
                    $invoices = Invoice::with('payments')->whereBetween('created_at', array($from, $to))->get();
                } else {
                    $invoices = $this->invoiceRepo->with(['payments'])->findAll(['*']);
                }
            }
            return DataTables::of($invoices)
                ->editColumn('serial', function ($invoice){
                    return "<a style='border: 1px solid; padding:2px;' href='".route('invoice.view', $invoice->id)."'>".$invoice->serial."</a>";
                })
                ->addColumn('client', function ($invoice){
                    return $invoice->client->name;
                })
                ->addColumn('total_paid', function ($invoice) {
                    return $invoice->payments->sum('paid_amount');
                })
                ->editColumn('grand_total', function ($invoice) {
                    return number_format($invoice->grand_total, 2);
                })
                ->editColumn('create_date', function ($invoice) {
                    return $invoice->create_date ? with(new Carbon($invoice->create_date))->format('jS F Y ') : '';
                })
                ->editColumn('payment_status', function ($invoice) {
                    if($invoice->payment_status == "unpaid") {
                        $btn = "btn-danger";
                    } elseif ($invoice->payment_status == "paid") {
                        $btn = "btn-success";
                    } else {
                        $btn = "btn-warning";
                    }
                    return "<button class='btn $btn btn-xs'>".$invoice->payment_status ."</button>";
                })
                ->editColumn('due_date', function ($invoice) {
                    return $invoice->due_date ? with(new Carbon($invoice->due_date))->format('jS F Y ') : '';
                })
                ->make(true);
        } else {
            return redirect()->route('report.list');
        }
    }

    public function paymentReportDataTable(ReportRequest $request)
    {
        if($request) {
            $client = $request->client_id;
            $from   = $request->from_date;
            $to     = $request->to_date;

            if ($request->has('client_id')) {
                if ($from && $to) {
                    $payments = Payment::with('invoice')->where('invoices.client_id',$client)
                                ->whereBetween('created_at', array($from, $to))->get();
                } elseif ($from && $to==null) {
                    $payments = Payment::with('invoice')->where('invoices.client_id',$client)
                        ->whereBetween('created_at', array($from, date('Y-m-d')))->get();
                } elseif ($to && $from==null) {
                    $payments = Payment::with('invoice')->where('invoices.client_id',$client)
                        ->whereBetween('created_at', array($from, $to))->get();
                } else {
                    $payments = Payment::with('invoice')->where('invoices.client_id', $client);
                }
            } else {
                if ($from && $to) {
                    $payments = Payment::with('invoice')->whereBetween('created_at', array($from, $to))->get();
                } elseif ($from && $to==null) {
                    $payments = Payment::with('invoice')->whereBetween('created_at', array($from, date('Y-m-d')))->get();
                } elseif ($to && $from==null) {
                    $payments = Payment::with('invoice')->whereBetween('created_at', array($from, $to))->get();
                } else {
                    $payments = Payment::with('invoice')->get();
                }
            }
            return DataTables::of($payments)
                ->editColumn('serial', function ($payment){
                    return "<a style='border: 1px solid; padding:2px;' href='".route('invoice.view', $payment->invoice->id)."'>".$payment->invoice->serial."</a>";
                })
                ->addColumn('user', function ($payment){
                    return $payment->user->name;
                })
                ->editColumn('paid_date', function ($payment) {
                    return $payment->paid_date ? with(new Carbon($payment->paid_date))->format('jS F Y ') : '';
                })
                ->make(true);
        } else {
            return redirect()->route('payment.report.list');
        }
    }

    public function expenseReportDataTable(ReportRequest $request)
    {
        if($request) {
            $user = $request->user_id;
            $from   = $request->from_date;
            $to     = $request->to_date;

            if ($request->has('user_id')) {
                if ($from && $to) {
                    $expenses = Expense::where('user_id',$user)
                                ->whereBetween('created_at', array($from, $to))->get();
                } elseif ($from && $to==null) {
                    $expenses = Expense::where('user_id',$user)
                        ->whereBetween('created_at', array($from, date('Y-m-d')))->get();
                } elseif ($to && $from==null) {
                    $expenses = Expense::where('user_id',$user)
                        ->whereBetween('created_at', array($from, $to))->get();
                } else {
                    $expenses = $this->expenseRepo->findWhere(['user_id', '=', $user]);
                }
            } else {
                if ($from && $to) {
                    $expenses = Expense::whereBetween('created_at', array($from, $to))->get();
                } elseif ($from && $to==null) {
                    $expenses = Expense::whereBetween('created_at', array($from, date('Y-m-d')))->get();
                } elseif ($to && $from==null) {
                    $expenses = Expense::whereBetween('created_at', array($from, $to))->get();
                } else {
                    $expenses = $this->expenseRepo->findAll(['*']);
                }
            }
            return DataTables::of($expenses)
                ->addColumn('purchased', function ($expense){
                    return $expense->purchased->name;
                })
                ->editColumn('purchased_date', function ($expense){
                    return $expense->purchased_date ? with(new Carbon($expense->purchased_date))->format('jS F Y') : '';
                })
                ->editColumn('expense_type_id', function ($expense){
                    return $expense->expenseType->name;
                })
                ->editColumn('payment_type_id', function ($expense){
                    return $expense->paymentType->name;
                })
                ->make(true);
        } else {
            return redirect()->route('expense.report.list');
        }
    }

    public function removeCached()
    {

    }


}
