<?php
/*
| -------------------------------------------------------------
| AUTHOR:           INILABS TEAM
| -------------------------------------------------------------
| EMAIL:            info@inilabs.net or labs.ini@gmail.com
| -------------------------------------------------------------
| WEBSITE:          http://inilabs.net
| -------------------------------------------------------------
*/
namespace Modules\Currency\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Modules\Currency\Models\Currency;
use Modules\Currency\Repositories\CurrencyRepository;
use Modules\Currency\Requests\CurrencyRequest;
use Yajra\Datatables\Facades\Datatables;
use Alert, Cache;

class CurrencyController extends Controller
{
    protected $currencyRepo;

    function __construct(CurrencyRepository $currency)
    {
        $this->currencyRepo = $currency;
    }

    public function langUses()
    {
        return view("Currency::index");
    }

    public function index()
    {
        return view("Currency::currencyList");
    }

    public function currencyDataTable()
    {
        return DataTables::of($this->currencyRepo->findAll(['*']))
            ->addColumn('actions', function ($currency){
                return $currency->action_buttons;
            })
            ->make(true);
    }

    public function add()
    {
        return view("Currency::currencyCreate");
    }

    public function create(CurrencyRequest $request)
    {
        $input = $request->all();
        $input['user_id'] = auth()->user()->id;
        $result = $this->currencyRepo->create($input);

        if ($result) {
            $this->removeCached();
            Alert::success(trans("Currency::currency.alert.update_success"), 'success');
            return redirect()->route('currency.list');
        }
    }

    public function edit(Currency $currency)
    {
        return view("Currency::currencyEdit")->withCurrency($currency);
    }

    public function update(Currency $currency, CurrencyRequest $request)
    {
        $result = $this->currencyRepo->update($currency, $request->all());

        if ($result) {
            $this->removeCached();
            Alert::success(trans("Currency::currency.alert.update_success"), 'success');
            return redirect()->route('currency.list');
        }
    }

    public function view(Currency $currency)
    {
        return $currency != null ? view("Currency::currencyView")->withCurrency($currency) :  abort(404);
    }

    public function status(Currency $currency)
    {
        $data = [];
        if ($currency->status==1) {
            $data['status'] = 0;
        } else {
            $data['status'] = 1;
        }
        $this->currencyRepo->update($currency->id, $data);
        $this->removeCached();
        Alert::success(trans("Currency::currency.alert.status_success"), 'success');
        return redirect()->route('currency.list');
    }

    public function delete(Currency $currency)
    {
        $this->currencyRepo->delete($currency->id);

        $this->removeCached();
        Alert::success(trans("Currency::currency.alert.delete_success"), 'success');
        return redirect()->route('currency.list');
    }

    public function removeCached()
    {
        Cache::forget('currencies');
    }


}
