<?php
/*
| -------------------------------------------------------------
| AUTHOR:           INILABS TEAM
| -------------------------------------------------------------
| EMAIL:            info@inilabs.net or labs.ini@gmail.com
| -------------------------------------------------------------
| WEBSITE:          http://inilabs.net
| -------------------------------------------------------------
*/
namespace Modules\Invoice\Controllers;

use App\Http\Controllers\Controller;
use Barryvdh\DomPDF\Facade as PDF;
use Carbon\Carbon;
use DaveJamesMiller\Breadcrumbs\View;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Modules\Bank\Models\Bank;
use Modules\Bank\Models\BankPayment;
use Modules\Client\Models\Client;
use Modules\Currency\Models\Currency;
use Modules\EmailTemplate\Models\EmailTemplate;
use Modules\EmailTemplate\Models\Tag;
use Modules\Invoice\Models\Payment;
use Modules\Iauth\Models\Access\User\User;
use Modules\Iauth\Repositories\Access\User\UserRepositoryContract;
use Modules\Invoice\Models\Invoice;
use Modules\Invoice\Models\OrderItem;
use Modules\Invoice\Repositories\InvoiceRepository;
use Modules\Invoice\Repositories\OrderItemRepository;
use Modules\Invoice\Requests\InvoiceRequest;
use Modules\Invoice\Requests\PaymentRequest;
use Modules\Invoice\Requests\PaymentSettingsRequest;
use Modules\PaymentType\Models\PaymentType;
use Modules\Product\Models\Product;
use Modules\Settings\Models\Settings;
use Modules\Tax\Models\Tax;
use Yajra\Datatables\Facades\Datatables;
use Alert, Cache;
use Omnipay\Omnipay;
use Illuminate\Support\Facades\Session;


class InvoiceController extends Controller
{
    protected $invoiceRepo;
    protected $userRepo;
    protected $orderItemRepo;


    function __construct(InvoiceRepository $invoice, UserRepositoryContract $client, OrderItemRepository $item)
    {
        $this->invoiceRepo = $invoice;
        $this->userRepo = $client;
        $this->orderItemRepo = $item;
    }

    public function __call($method,$arguments) {
        $paymentOption = 'paymentOption'.ucfirst($method);
        if(method_exists($this,$paymentOption)) {
            return call_user_func_array([$this,$paymentOption],$arguments);
        } else {
            $paymentOption = 'paymentOptionOverAll';
            return call_user_func_array([$this,$paymentOption],$arguments);
        }
    }

    public function index()
    {
        return view("Invoice::invoiceList");
    }
    public function paymentsList()
    {
        return view("Invoice::paymentList");
    }
    public function invoiceDataTable()
    {
        if(auth()->user()->hasrole(1)) {
            $invoices = $this->invoiceRepo->orderBy('id', 'desc')->findAll();
        } else {
            $invoices = $this->invoiceRepo->orderBy('id', 'desc')->findWhere(['client_id', '=', auth()->user()->id]);
        }
        return DataTables::of($invoices)
            ->editColumn('serial', function ($invoice){
                return "<a style='border: 1px solid; padding:2px;' href='".route('invoice.view', $invoice->id)."'>".$invoice->serial."</a>";
            })
            ->addColumn('client', function ($invoice){
                  return $invoice->client->name;
            })
            ->addColumn('actions', function ($invoice){
                return $invoice->action_buttons;
            })
            ->editColumn('create_date', function ($invoice) {
                return $invoice->create_date ? with(new Carbon($invoice->create_date))->format('jS F Y ') : '';
            })
            ->editColumn('grand_total', function ($invoice){
                return $invoice->currency->currency_symbol ? $invoice->currency->currency_symbol.number_format($invoice->grand_total, 2) : isite()->siteCurrencySymbol().number_format($invoice->grand_total, 2);
            })
            ->editColumn('payment_status', function ($invoice) {
                if($invoice->payment_status == "unpaid") {
                    $btn = "btn-danger";
                } elseif ($invoice->payment_status == "paid") {
                    $btn = "btn-success";
                } else {
                    $btn = "btn-warning";
                }
                return "<button class='btn $btn btn-xs'>".$invoice->payment_status ."</button>";
            })
            ->editColumn('due_date', function ($invoice) {
                return $invoice->due_date ? with(new Carbon($invoice->due_date))->format('jS F Y ') : '';
            })
            ->make(true);
    }

    public function paymentsListDataTable()
    {
        if(auth()->user()->hasrole(1)) {
            $payments = Payment::with('invoice')->orderBy('paid_date', 'desc')->get();
        } else {
            $payments = Payment::with('invoice')->where(['user_id' => auth()->user()->id])->orderBy('paid_date', 'desc')->get();
        }

        return DataTables::of($payments)
            ->editColumn('serial', function ($payment){
                return "<a style='border: 1px solid; padding:2px;' href='".route('invoice.view', $payment->invoice->id)."'>".$payment->invoice->serial."</a>";
            })
            ->addColumn('user', function ($payment){
                  return $payment->user->name;
            })
            ->editColumn('paid_date', function ($payment) {
                return $payment->paid_date ? with(new Carbon($payment->paid_date))->format('jS F Y ') : '';
            })
            ->editColumn('paid_amount', function ($payment){
                return isite()->siteCurrencySymbol().number_format($payment->paid_amount, 2);
            })
            ->make(true);
    }

    public function add()
    {
        $client         = Cache::remember('all_clients',60, function () {
            return $this->userRepo->getAllUsersByRole('Client')->pluck('username', 'id')->toArray();
        });

        $taxes    = Cache::remember('taxes',60, function () {
            return Tax::all()->pluck('title', 'id')->toArray();
        });

        $currencies    = Cache::remember('currencies',60, function () {
            return Currency::all()->pluck('currency_code', 'id')->toArray();
        });

        $products = Cache::remember('products',60, function () {
            return Product::all()->pluck('name', 'amount')->toArray();
        });

        $payment_types = Cache::remember('payment_types',60, function () {
            return PaymentType::where('status', 1)->pluck('name', 'id')->toArray();
        });
        return view("Invoice::invoiceCreate")->withClients($client)->withTaxes($taxes)->withProducts($products)->withPaymentTypes($payment_types)->withCurrencies($currencies);
    }
    public function store(InvoiceRequest $request)
    {
        $create_date = new \DateTime($request->create_date);
        $due_date = new \DateTime($request->due_date);
        $request['create_date'] = $create_date->format('Y-m-d');
        $request['due_date'] = $due_date->format('Y-m-d');

        if ($request->tax_id == "") {
            unset($request['tax_id']);
        }
        $products = collect($request->products)->transform(function($product) {
            $product['total'] = $product['quantity'] * $product['price'];
            return new OrderItem($product);
        });

        if($products->isEmpty()) {
            return response()
                ->json([
                    'products_empty' => ['One or more Product is required.']
                ], 422);
        }

        $data = $request->except('products', 'action');
        $data['sub_total'] = $products->sum('total');
        $tax_amount = 0;
        if ($request->has('tax_id')) {
            $tax = Tax::findOrFail($request->tax_id);
            if ($tax->type==1) {
                $percentage = $tax->rate;
                $subTotal = $data['sub_total'];
                $tax_amount = ($percentage / 100) * $subTotal;
            } elseif($tax->type==2) {
                $tax_amount = $tax->rate;
            }
        }
        $data['grand_total'] = ($data['sub_total'] - $data['discount']) + $tax_amount;
        $data['user_id'] = auth()->user()->id;
        $data['token'] = "token_".mt_rand(100000, 999999);

        $invoice = Invoice::create($data);

        $invoice->products()->saveMany($products);

        if ($request->action=="send") {
            $this->sendInvoice($invoice);
        }
        $this->removeCached();

        Alert::success(trans("Invoice::invoice.alert.create_success"), 'success');

        return response()
            ->json([
                'created' => true,
                'id' => $invoice->id
            ]);
    }

    public function edit($id)
    {
        $client         = Cache::remember('all_clients',60, function () {
            return $this->userRepo->getAllUsersByRole('Client')->pluck('username', 'id')->toArray();
        });

        $taxes    = Cache::remember('taxes',60, function () {
            return Tax::all()->pluck('title', 'id')->toArray();
        });

        $products = Cache::remember('products',60, function () {
            return Product::all()->pluck('name', 'amount')->toArray();
        });

        $currencies    = Cache::remember('currencies',60, function () {
            return Currency::all()->pluck('currency_code', 'id')->toArray();
        });

        $payment_types = Cache::remember('payment_types',60, function () {
            return PaymentType::where('status', 1)->pluck('name', 'id')->toArray();
        });

        $invoice = Invoice::with('products')->findOrFail($id);
        $create_date = new \DateTime($invoice->create_date);
        $due_date = new \DateTime($invoice->due_date);
        $invoice->create_date = $create_date->format('Y-m-d');
        $invoice->due_date = $due_date->format('Y-m-d');

        return view('Invoice::invoiceEdit')->withClients($client)->withTaxes($taxes)->withInvoice($invoice)->withPaymentTypes($payment_types)->withProducts($products)->withCurrencies($currencies);
    }

    public function update(InvoiceRequest $request, $id)
    {
        $create_date = new \DateTime($request->create_date);
        $due_date = new \DateTime($request->due_date);
        $request['create_date'] = $create_date->format('Y-m-d');
        $request['due_date'] = $due_date->format('Y-m-d');

        $invoice = Invoice::findOrFail($id);

        $products = collect($request->products)->transform(function($product) {
            $product['total'] = $product['quantity'] * $product['price'];
            return new OrderItem($product);
        });

        if($products->isEmpty()) {
            return response()
                ->json([
                    'products_empty' => ['One or more Product is required.']
                ], 422);
        }

        $data = $request->except('products', 'action');
        $data['sub_total'] = $products->sum('total');
        $tax_amount = 0;
        if ($request->has('tax_id')) {
            $tax = Tax::findOrFail($request->tax_id);
            if ($tax->type==1) {
                $percentage = $tax->rate;
                $subTotal = $data['sub_total'];
                $tax_amount = ($percentage / 100) * $subTotal;
            } elseif($tax->type==2) {
                $tax_amount = $tax->rate;
            }
        }
        $data['grand_total'] = ($data['sub_total'] - $data['discount']) + $tax_amount;

        if ($request->action=="send") {
            $this->sendInvoice($invoice);
        }
        $invoice->update($data);

        OrderItem::where('invoice_id', $invoice->id)->delete();

        $invoice->products()->saveMany($products);

        $this->removeCached();

        return response()
            ->json([
                'updated' => true,
                'id' => $invoice->id,
                'url' => route('invoice.view', $invoice->id)
            ]);
    }
    public function view($id)
    {
        $this->removeCached();
        $invoice   = Cache::remember('invoice',60, function () use ($id) {
            return Invoice::with('products')->with('payments.user')->findOrFail($id);
        });

        $email_templates   = Cache::remember('email_templates',60, function () use ($id) {
            return EmailTemplate::where('status', 1)->pluck('name', 'id')->toArray();
        });

        $payment_types = Cache::remember('payment_types',60, function () {
            return PaymentType::where('status', 1)->pluck('name', 'id')->toArray();
        });

        $banks = Cache::remember('banks',60, function () {
            return Bank::where('status', 1)->pluck('bank_title', 'id')->toArray();
        });

        $client = User::findOrFail($invoice->client_id);
        $relationTableData = $client->hasOneRelation('Modules\Client\Models\Client')->first();

        if (isset($invoice->tax_id)) {
            $tax = Tax::find($invoice->tax_id);
            return view("Invoice::invoiceView", compact('email_templates'))->withInvoice($invoice)->withTax($tax)->withClient($client)->withCompany($relationTableData)->withPaymentTypes($payment_types)->withBanks($banks);
        }
        return view("Invoice::invoiceView",  compact('email_templates'))->withInvoice($invoice)->withClient($client)->withCompany($relationTableData)->withPaymentTypes($payment_types)->withBanks($banks);

    }

    public function delete($id)
    {
        $invoice = Invoice::findOrFail($id);
        OrderItem::where('invoice_id', $invoice->id)
            ->delete();
        $invoice->delete();
        $this->removeCached();
        return redirect()
            ->route('invoice.list');
    }

    /*
     * Payment Process
     */
    public function getPaypalCancel(Invoice $invoice)
    {
        Alert::error("error", 'Payment Not Successful!');
        return redirect()->route('invoice.view', $invoice);
    }

    public function getPaypalSuccess(Invoice $invoice)
    {
        $payment_option =  Settings::all()->pluck('value', 'option')->toArray();

        $gateway = Omnipay::create('PayPal_Express');
        $gateway->setUsername($payment_option['payment_paypal_username']);
        $gateway->setPassword($payment_option['payment_paypal_password']);
        $gateway->setSignature($payment_option['payment_paypal_signature']);
        $gateway->setTestMode($payment_option['payment_sandbox_status']);

        $params = Session::get('params');
        $response = $gateway->completePurchase($params)->send();
        $paypalResponse = $response->getData(); // this is the raw response object

        if(isset($paypalResponse['PAYMENTINFO_0_ACK']) && $paypalResponse['PAYMENTINFO_0_ACK'] === 'Success') {
            // Response
            $payment = Payment::where('transaction_id',$paypalResponse['PAYMENTINFO_0_TRANSACTIONID'])->first();
            if($payment){
                Alert::warning("warning", 'Transaction ID Already Exist');
                return redirect()->route('invoice.view', $invoice);
            } else{
                $payment=new Payment;
                $payment->invoice_id=$invoice->id;
                $payment->transaction_id=$paypalResponse['PAYMENTINFO_0_TRANSACTIONID'];
                $payment->currency_code=$paypalResponse['PAYMENTINFO_0_CURRENCYCODE'];
                $payment->payment_status=$paypalResponse['PAYMENTINFO_0_PAYMENTSTATUS'];
                $payment->paid_amount=$paypalResponse['PAYMENTINFO_0_AMT'];
                $payment->user_id = auth()->user()->id;
                $payment->payment_type = "paypal";
                $payment->save();
                $invoice = Invoice::with('payments')->findOrFail($payment->invoice_id);
                $totalPaid = $invoice->payments->sum('paid_amount');
                if ($totalPaid >= $invoice->grand_total ) {
                    $data['payment_status'] = "paid";
                } elseif($totalPaid <= $invoice->grand_total ) {
                    $data['payment_status'] = "partially paid";
                } else {
                    $data['payment_status'] = "pending";
                }
                $invoice->update($data);
                Alert::success("success", 'Payment Successful!');
                return redirect()->route('invoice.view', $invoice);
            }
        } else {
            Alert::error("error", 'Payment Not Successful!');
            return redirect()->route('invoice.view', $invoice);
            //Failed transaction
        }
    }

    public function getTwoCheckoutCancel(Invoice $invoice)
    {
        Alert::error("error", 'Payment Not Successful!');
        return redirect()->route('invoice.view', $invoice);
    }

    public function getTwoCheckoutSuccess(Invoice $invoice)
    {
        $payment_option =  Settings::all()->pluck('value', 'option')->toArray();

        $gateway = Omnipay::create('TwoCheckout');
        $gateway->setSecretWord($payment_option['payment_twocheckout_secret_word']);
        $gateway->setAccountNumber($payment_option['payment_twocheckout_account']);
        $gateway->setTestMode($payment_option['payment_sandbox_status']); // turns on Sandbox access

        $params = Session::get('params');
        $response = $gateway->completePurchase($params)->send();
        $responseData = $response->getData(); // this is the raw response object

        if(isset($responseData['PAYMENTINFO_0_ACK']) && $responseData['PAYMENTINFO_0_ACK'] === 'Success') {
            // Response
            $payment = Payment::where('transaction_id',$responseData['PAYMENTINFO_0_TRANSACTIONID'])->first();
            if($payment){
                Alert::warning("warning", 'Transaction ID Already Exist');
                return redirect()->route('invoice.view', $invoice);
            } else{
                $payment=new Payment;
                $payment->invoice_id=$invoice->id;
                $payment->transaction_id=$responseData['PAYMENTINFO_0_TRANSACTIONID'];
                $payment->currency_code=$responseData['PAYMENTINFO_0_CURRENCYCODE'];
                $payment->payment_status=$responseData['PAYMENTINFO_0_PAYMENTSTATUS'];
                $payment->paid_amount=$responseData['PAYMENTINFO_0_AMT'];
                $payment->user_id = auth()->user()->id;
                $payment->payment_type = "paypal";
                $payment->save();
                $invoice = Invoice::with('payments')->findOrFail($payment->invoice_id);
                $totalPaid = $invoice->payments->sum('paid_amount');
                if ($totalPaid >= $invoice->grand_total ) {
                    $data['payment_status'] = "paid";
                } elseif($totalPaid <= $invoice->grand_total ) {
                    $data['payment_status'] = "partially paid";
                } else {
                    $data['payment_status'] = "pending";
                }
                $invoice->update($data);
                Alert::success("success", 'Payment Successful!');
                return redirect()->route('invoice.view', $invoice);
            }
        } else {
            Alert::error("error", 'Payment Not Successful!');
            return redirect()->route('invoice.view', $invoice);
            //Failed transaction
        }
    }


    public function payment(Invoice $invoice, PaymentRequest $request)
    {
        $paymentOption = $request->payment_option;
        $this->$paymentOption($invoice, $request);
        return redirect()->route('invoice.view', $invoice);
    }
    // Online Payment Options

    public function paymentOptionOverAll($invoice, $request)
    {
        if ($request->has('payment_option')) {
            $payment_option =  Settings::all()->pluck('value', 'option')->toArray();
            $payment=new Payment;
            $payment->invoice_id=$invoice->id;
            $payment->transaction_id= str_random(20);
            $payment->currency_code= $payment_option['site_currency_code'];
            $payment->payment_status= "Completed";
            $payment->paid_amount= $request->paying_amount;
            $payment->paid_date= $request->paid_date;
            $payment->user_id = auth()->user()->id;
            $payment->payment_type = $request->payment_option;
            $payment->save();
            $invoice = Invoice::with('payments')->findOrFail($payment->invoice_id);
            $totalPaid = $invoice->payments->sum('paid_amount');
            if ($totalPaid >= $invoice->grand_total ) {
                $data['payment_status'] = "paid";
            } elseif($totalPaid <= $invoice->grand_total ) {
                $data['payment_status'] = "partially paid";
            } else {
                $data['payment_status'] = "pending";
            }
            $invoice->update($data);
            Alert::success("success", 'Payment Successful!');
            return redirect()->route('invoice.view', $invoice);

        } else {
            Alert::error("error", 'Select Payment Option');
            return redirect()->route('invoice.view', $invoice);
        }
    }
    public function paymentOptionBank($invoice, $request)
    {
        if ($request->payment_option == 'bank') {
            if ($request->has('bank_id')) {
                $payment_option =  Settings::all()->pluck('value', 'option')->toArray();
                $payment=new Payment;
                $payment->invoice_id=$invoice->id;
                $payment->transaction_id= str_random(20);
                $payment->currency_code= $payment_option['site_currency_code'];
                $payment->payment_status= "Completed";
                $payment->paid_amount= $request->paying_amount;
                $payment->paid_date= $request->paid_date;
                $payment->user_id = auth()->user()->id;
                $payment->payment_type = $request->payment_option;
                $payment->save();
                $payment_id = $payment->id;

                $bank_payment = new BankPayment();
                $bank_payment->bank_id = $request->bank_id;
                $bank_payment->payment_id = $payment_id;
                $bank_payment->save();

                $invoice = Invoice::with('payments')->findOrFail($payment->invoice_id);
                $totalPaid = $invoice->payments->sum('paid_amount');
                if ($totalPaid >= $invoice->grand_total ) {
                    $data['payment_status'] = "paid";
                } elseif($totalPaid <= $invoice->grand_total ) {
                    $data['payment_status'] = "partially paid";
                } else {
                    $data['payment_status'] = "pending";
                }
                $invoice->update($data);
                Alert::success("success", 'Payment Successful!');
                return redirect()->route('invoice.view', $invoice);

            } else {
                Alert::error("error", "Something gone wrong! please select bank account");
                return redirect()->route('invoice.view', $invoice);
            }
        } else {
            Alert::error("error", 'Select Payment Option');
            return redirect()->route('invoice.view', $invoice);
        }
    }
    public function paymentOptionPaypal($invoice, $request)
    {
        if ($request->payment_option == 'paypal') {
            $payment_option =  Settings::all()->pluck('value', 'option')->toArray();
            $params = array(
                'cancelUrl'     => route('invoice.paypal.cancel', $invoice->id),
                'returnUrl'     => route('invoice.paypal.success', $invoice->id),
                'item_number'   => $invoice->id,
                'serial'        => $invoice->serial,
                'amount'        => (float)$request->paying_amount,
                'currency'      => $invoice->currency->currency_code ? $invoice->currency->currency_code : $payment_option['site_currency_code']
            );

            Session::put('params', $params);
            Session::save();
            $gateway = Omnipay::create('PayPal_Express');
            $gateway->setUsername($payment_option['payment_paypal_username']);
            $gateway->setPassword($payment_option['payment_paypal_password']);
            $gateway->setSignature($payment_option['payment_paypal_signature']);
            $gateway->setTestMode($payment_option['payment_sandbox_status']);
            $response = $gateway->purchase($params)->send();


            if ($response->isSuccessful()) {
                // payment was successful: update database
                print_r($response);
            } elseif ($response->isRedirect()) {
                // redirect payment gateway
                $response->redirect();
            } else {
                // payment failed: display message to customer
                echo $response->getMessage();
            }
        } else {
            Alert::error("error", 'Select Payment Option');
            return redirect()->route('invoice.view', $invoice);
        }
    }
    public function paymentOptionStripe($invoice, $request)
    {
        if ($request->payment_option == 'stripe') {
            if ($request->has('card_number') && $request->has('cvv') && $request->has('expire_month') && $request->has('expire_year')) {
                $payment_option =  Settings::all()->pluck('value', 'option')->toArray();
                try {
                    $gateway = Omnipay::create('Stripe');
                    $gateway->setApiKey($payment_option['payment_stripe_secret_key']);

                    $formData = array('number' => $request->card_number, 'expiryMonth' => $request->expire_month, 'expiryYear' => $request->expire_year, 'cvv' => $request->cvv);
                    $response = $gateway->purchase(array(
                            'amount' => $request->paying_amount,
                            'currency' => $payment_option['site_currency_code'],
                            'card' => $formData)
                    )->send();

                    if ($response->isSuccessful()) {
                        // payment was successful: update database
                        if($response->getData()['status']==="succeeded") {
                            $payment=new Payment;
                            $payment->invoice_id=$invoice->id;
                            $payment->transaction_id= $response->getData()['id'];
                            $payment->currency_code= $payment_option['site_currency_code'];
                            $payment->payment_status= "Completed";
                            $payment->paid_amount= $request->paying_amount;
                            $payment->user_id = auth()->user()->id;
                            $payment->payment_type = $request->payment_option;
                            $payment->save();
                            $invoice = Invoice::with('payments')->findOrFail($payment->invoice_id);
                            $totalPaid = $invoice->payments->sum('paid_amount');
                            if ($totalPaid >= $invoice->grand_total ) {
                                $data['payment_status'] = "paid";
                            } elseif($totalPaid <= $invoice->grand_total ) {
                                $data['payment_status'] = "partially paid";
                            } else {
                                $data['payment_status'] = "pending";
                            }
                            $invoice->update($data);
                            Alert::success("success", 'Payment Successful!');
                            return redirect()->route('invoice.view', $invoice);
                        }
                    } elseif ($response->isRedirect()) {
                        // redirect to offsite payment gateway
                        $response->redirect();
                    } else {
                        // payment failed: display message to customer
                        Alert::error("error", "Something gone wrong! your payment is not successful");
                        return redirect()->route('invoice.view', $invoice);
                    }
                } catch(\Exception $ex){
                    Alert::error("error", $ex->getMessage());
                    return redirect()->route('invoice.view', $invoice);
                }
            } else {
                Alert::error("error", "Something gone wrong! card information is not valid");
                return redirect()->route('invoice.view', $invoice);
            }
        }
    }
    public function paymentOptionTwocheckout($invoice, $request)
    {
        if ($request->payment_option == 'twocheckout') {
            $payment_option =  Settings::all()->pluck('value', 'option')->toArray();
            try {
                $MerchantTransID = rand(11111111,99999999);
                $params = array(
                    'cancelUrl'     => route('invoice.twocheckout.cancel', $invoice->id),
                    'returnUrl'     => route('invoice.twocheckout.success', $invoice->id),
                    'item_number'   => $invoice->id,
                    'serial'        => $invoice->serial,
                    'amount'        => (float)$request->paying_amount,
                    'currency'      => $payment_option['site_currency_code'],
                    'transactionId' => $MerchantTransID,
                    "sellerId" => "102638836",
                    "merchantOrderId" => "123",
                    "token" => 'Y2U2OTdlZjMtOGQzMi00MDdkLWJjNGQtMGJhN2IyOTdlN2Ni',
                );

                Session::put('params', $params);
                Session::save();

                $gateway = Omnipay::create('TwoCheckout');
                $gateway->setSecretWord($payment_option['payment_twocheckout_secret_word']);
                $gateway->setAccountNumber($payment_option['payment_twocheckout_account']);
                $gateway->setTestMode($payment_option['payment_sandbox_status']); // turns on Sandbox access

                $response = $gateway->purchase($params)->send();

                if ($response->isSuccessful()) {
                    dd($response->getData());
                } elseif ($response->isRedirect()) {
                    // redirect to off site payment gateway
                    $response->redirect();
                } else {
                    // payment failed: display message to customer
                    echo $response->getMessage();
                }
            } catch(\Exception $ex){
                Alert::error("error", $ex->getMessage());
                return redirect()->route('invoice.view', $invoice);
            }
        }
    }
    public function paymentOptionPayumoney($invoice, $request)
    {
        if ($request->payment_option == 'payumoney') {
            $payment_option =  Settings::all()->pluck('value', 'option')->toArray();
            $merchantId = $payment_option['payment_payumoney_merchantID'];
            $secretKey = $payment_option['payment_payumoney_secretkey'];
            try {
                $PayU = Omnipay::create('PayUBiz');
                $PayU->setKey('gtKFFx');
                $PayU->setSalt('eCwWELxi');
                $PayU->setTestMode(true);

                $PaymentDetails = [
                    'transactionId' => 12345678,
                    'amount' => 4000.00,
                    'product' => "Pen",
                    'firstName' => "Rid",
                    'name' => "Islam",
                    'email' => "i.ridislam@gmail.com",
                    'returnUrl' => url('api/payment/success'),
                    'curl' => url('api/payment/cancelled'),
                    'furl' => url('api/payment/failed'),
                ];

                $PayU->purchase($PaymentDetails)->send()->redirect();

            } catch(\Exception $ex){
                Alert::error("error", $ex->getMessage());
                return redirect()->route('invoice.view', $invoice);
            }
        }
    }
    public function getPayUCancel(Invoice $invoice)
    {
        Alert::error("error", 'Payment Not Successful!');
        return redirect()->route('invoice.view', $invoice);
    }
    public function getPayUSuccess(Invoice $invoice, Request $request)
    {
        dd("ok");
    }
    //send invoice email
    public function sendEmail($id, Request $request)
    {
        if(getenv('MAIL_USERNAME')!=null && getenv('MAIL_PASSWORD') != null) {
            $this->validate($request, [
                'email' => 'required|email',
                'subject' => 'required',
                'body' => 'required',
            ]);
            $attachedFileName = '';
            $title      = "<h2>$request->subject</h2>";
            $content    = "<p>$request->body</p>";
            if ($request->has('attachStatus')) {
                $attachedFileName = $this->storeInvoice($id);
            }

            Mail::send('emails.send', ['title' => $title, 'content' => $content], function ($message) use ($request, $attachedFileName)
            {
                $message->from(isite()->siteEmail(), isite()->siteName());
                $message->to($request->email);
                if ($attachedFileName!="") {
                    $message->attach($attachedFileName, ['mime' => 'pdf']);
                }
            });
            $data = [
                'success' => true,
            ];
            return response()->json($data);
        } else {
            $data = [
                'error' => "Please Set Smtp Information!",
            ];
            return response()->json($data);
        }
    }
    public function sendInvoice($invoice)
    {
        if(getenv('MAIL_USERNAME')!=null && getenv('MAIL_PASSWORD') != null) {

            $title      = "<h2>Invoice $invoice->serial</h2>";
            $content    = "<p>Dear ".$invoice->client->username.", Please check attachment.</p>";
            $attachedFileName = $this->storeInvoice($invoice->id);

            Mail::send('emails.send', ['title' => $title, 'content' => $content], function ($message) use ($invoice, $attachedFileName)
            {
                $message->from(isite()->siteEmail(), isite()->siteName());
                $message->to($invoice->client->email);
                if ($attachedFileName!="") {
                    $message->attach($attachedFileName, ['mime' => 'pdf']);
                }
            });
            $data = [
                'success' => true,
            ];
            return response()->json($data);
        } else {
            $data = [
                'error' => "Please Set Smtp Information!",
            ];
            return response()->json($data);
        }
    }
    public function removeCached()
    {
        Cache::forget('invoices');
        Cache::forget('all_invoices');
        Cache::forget('invoice');
        Cache::forget('payments');
        Cache::forget('bankBalance');
        Cache::forget('all_invoices_'.auth()->user()->id);
    }

    public function updatePaymentSettings(PaymentSettingsRequest $request)
    {
        $inputs = array_except($request->all(), ['_method', '_token']);
        if ($request->has('payment_sandbox_status')) {
            $inputs['payment_sandbox_status'] = 'true';
        } else {
            $inputs['payment_sandbox_status'] = 'false';
        }

        $this->saveToDatabase($inputs);
        Artisan::call('cache:clear');
        Alert::success(trans("Settings::settings.alert.update_success"), 'success');
        return redirect(route('settings'))->withTab('payment');
    }

    public function saveToDatabase($inputs)
    {
        foreach ($inputs as $option => $value) {
            Settings::updateOrInsert(['option' => $option],['value' => $value]);
        }
    }

    public function storeInvoice($id)
    {
        $invoice   = Cache::remember('invoice',60, function () use ($id) {
            return Invoice::with('products')->with('payments.user')->findOrFail($id);
        });
        $payment_types = Cache::remember('payment_types',60, function () {
            return PaymentType::where('status', 1)->pluck('name', 'id')->toArray();
        });
        $tax = Tax::find($invoice->tax_id);
        $client = User::findOrFail($invoice->client_id);
        $company = $client->hasOneRelation('Modules\Client\Models\Client')->first();

        $pdf = PDF::loadView('Invoice::pdf', compact('invoice', 'payment_types', 'tax', 'client', 'company'));
        $invoicePath = public_path('assets/pdf/').date('Y-m-d-H-i-s')."-invoice-".$invoice->serial.'.pdf';
        if($pdf->save($invoicePath)) {
            return $invoicePath;
        } else {
            return false;
        }
    }

    public function invoicePdf($id)
    {
        $invoice   = Cache::remember('invoice',60, function () use ($id) {
            return Invoice::with('products')->with('payments.user')->findOrFail($id);
        });
        $payment_types = Cache::remember('payment_types',60, function () {
            return PaymentType::where('status', 1)->pluck('name', 'id')->toArray();
        });
        $tax = Tax::find($invoice->tax_id);
        $client = User::findOrFail($invoice->client_id);
        $company = $client->hasOneRelation('Modules\Client\Models\Client')->first();

        $pdf = PDF::loadView('Invoice::pdf', compact('invoice', 'payment_types', 'tax', 'client', 'company'));
        return $pdf->stream();
    }

    public function globalAccess($serial, $token)
    {
        $invoice   = Invoice::where('serial', '=', $serial)->where('token', '=', $token)->with('products')->with('payments.user')->first();
        $tax = Tax::find($invoice->tax_id);
        $client = User::findOrFail($invoice->client_id);
        $company = $client->hasOneRelation('Modules\Client\Models\Client')->first();

        return view("Invoice::globalPdf", compact('invoice', 'tax', 'client', 'company'));
    }
    public function globalPdf($serial, $token)
    {
        $invoice   = Invoice::where('serial', '=', $serial)->where('token', '=', $token)->with('products')->with('payments.user')->first();
        $tax = Tax::find($invoice->tax_id);
        $client = User::findOrFail($invoice->client_id);
        $company = $client->hasOneRelation('Modules\Client\Models\Client')->first();

        $pdf = PDF::loadView('Invoice::pdf', compact('invoice', 'payment_types', 'tax', 'client', 'company'));
        return $pdf->stream();
    }

    public function emailTemplate(Request $request)
    {
        $invoice    = Invoice::findOrFail($request->invoice_id);
        $template   = EmailTemplate::findOrFail($request->template_id);
        if(count($template)) {
            echo $this->translateEmail($template->template, $invoice);
        }
    }

    function translateEmail($message, $invoice) {
        if($invoice) {
            $Tags = Tag::all()->pluck('tag_name', 'id');
            $client = User::where('id', $invoice->client_id)->first();
            if(count($Tags)) {
                foreach ($Tags as $key => $tag) {
                    if($tag == '{name}') {
                        if($client->first_name) {
                            $message = str_replace('{name}', $client->first_name, $message);
                        } else {
                            $message = str_replace('{name}', ' ', $message);
                        }
                    } elseif($tag == '{email}') {
                        if(isite()->siteEmail()) {
                            $message = str_replace('{email}', isite()->siteEmail(), $message);
                        } else {
                            $message = str_replace('{email}', ' ', $message);
                        }
                    } elseif($tag == '{phone}') {
                        if($client->phone) {
                            $message = str_replace('{phone}', $client->phone, $message);
                        } else {
                            $message = str_replace('{phone}', ' ', $message);
                        }
                    } elseif($tag == '{address}') {
                        if($client->address) {
                            $message = str_replace('{address}', $client->address, $message);
                        } else {
                            $message = str_replace('{address}', ' ', $message);
                        }
                    } elseif($tag == '{username}') {
                        if($client->username) {
                            $message = str_replace('{username}', $client->username, $message);
                        } else {
                            $message = str_replace('{username}', ' ', $message);
                        }
                    } elseif($tag == '{country}') {
                        if($client->country) {
                            $message = str_replace("{country}", $client->country, $message);
                        } else {
                            $message = str_replace("{country}", ' ', $message);
                        }
                    } elseif($tag == '{state}') {
                        if($client->state) {
                            $message = str_replace("{state}", $client->state, $message);
                        } else {
                            $message = str_replace("{state}", ' ', $message);
                        }
                    } elseif($tag == '{invoice_url}') {
                        if($invoice) {
                            $message = str_replace("{invoice_url}", route('invoice.view', $invoice->id), $message);
                        } else {
                            $message = str_replace("{invoice_url}", ' ', $message);
                        }
                    } elseif($tag == '{invoice_serial}') {
                        if($invoice) {
                            $message = str_replace("{invoice_serial}", "$invoice->serial", $message);
                        } else {
                            $message = str_replace("{invoice_serial}", ' ', $message);
                        }
                    } elseif($tag == '{invoice_amount}') {
                        if($invoice) {
                            $message = str_replace("{invoice_amount}", "$invoice->grand_total", $message);
                        } else {
                            $message = str_replace("{invoice_amount}", ' ', $message);
                        }
                    } elseif($tag == '{invoice_due_date}') {
                        if($invoice->due_date) {
                            $message = str_replace("{invoice_due_date}", "$invoice->due_date", $message);
                        } else {
                            $message = str_replace("{invoice_due_date}", ' ', $message);
                        }
                    } elseif($tag == '{business_name}') {
                        if(isite()->siteName()) {
                            $message = str_replace("{business_name}", isite()->siteName(), $message);
                        } else {
                            $message = str_replace("{business_name}", ' ', $message);
                        }
                    } elseif($tag == '{business_email}') {
                        if(isite()->siteEmail()) {
                            $message = str_replace("{business_email}", isite()->siteEmail(), $message);
                        } else {
                            $message = str_replace("{business_email}", ' ', $message);
                        }
                    } elseif($tag == '{client_email}') {
                        if($client->email) {
                            $message = str_replace("{client_email}", "$client->email", $message);
                        } else {
                            $message = str_replace("{client_email}", ' ', $message);
                        }
                    } elseif($tag == '{login_url}') {
                        if($client) {
                            $message = str_replace("{login_url}", route('admin.login'), $message);
                        } else {
                            $message = str_replace("{login_url}", ' ', $message);
                        }
                    } elseif($tag == '{password_reset_link}') {
                        if($client) {
                            $message = str_replace("{password_reset_link}", route('admin.password.reset'), $message);
                        } else {
                            $message = str_replace("{password_reset_link}", ' ', $message);
                        }
                    } elseif($tag == '{logo}') {
                        if(get_logo()) {
                            $message = str_replace("{logo}", "<img src='".url(get_logo())."' height='50px' class='pull-left thumb-xs m-r-sm'>", $message);
                        } else {
                            $message = str_replace("{logo}", ' ', $message);
                        }
                    } elseif($tag == '{site_name}') {
                        if(isite()->siteName()) {
                            $message = str_replace("{site_name}", isite()->siteName(), $message);
                        } else {
                            $message = str_replace("{site_name}", ' ', $message);
                        }
                    } elseif($tag == '{site_link}') {
                        $message = str_replace("{site_link}", url(''), $message);
                    } elseif($tag == '{invoice_global_link}') {
                        $message = str_replace("{invoice_global_link}", route('invoice.global', [$invoice->serial,$invoice->token]), $message);
                    }
                }
            }

            return $message;
        }
    }
}
