<?php
/*
| -------------------------------------------------------------
| iBill - Simplest and Best Billing & Invoice Manager
| -------------------------------------------------------------
| AUTHOR:           INILABS TEAM
| -------------------------------------------------------------
| EMAIL:            info@inilabs.net
| -------------------------------------------------------------
| COPYRIGHT:        RESERVED BY INILABS IT
| -------------------------------------------------------------
| WEBSITE:          http://inilabs.net
| -------------------------------------------------------------
*/
namespace Modules\Notice\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Modules\Notice\Events\History\NoticeCreated;
use Modules\Notice\Events\History\NoticeDeleted;
use Modules\Notice\Events\History\NoticeUpdated;
use Modules\Notice\Models\Notice;
use Modules\Notice\Models\NoticeAlert;
use Modules\Notice\Repositories\NoticeRepository;
use Modules\Notice\Requests\NoticeRequest;
use Yajra\Datatables\Facades\Datatables;
use Alert, Cache;

class NoticeController extends Controller
{
    protected $noticeRepo;

    function __construct(NoticeRepository $notice)
    {
        $this->noticeRepo = $notice;
    }

    public function index()
    {
        return view("Notice::noticeList");
    }

    public function noticeDataTable()
    {
        return DataTables::of($this->noticeRepo->findAll(['*']))
            ->addColumn('users', function ($notice){
                return $notice->user->username;
            })
            ->editColumn('body', function ($notice){
                $lenth = strlen($notice->body);
                if ($lenth > 99)
                    return substr($notice->body, 0, 100)."...";
                else
                    return $notice->body;
            })
            ->addColumn('actions', function ($notice){
                return $notice->action_buttons;
            })
            ->make(true);
    }

    public function add()
    {
        return view("Notice::noticeCreate");
    }

    public function create(NoticeRequest $request)
    {
        $input = $request->all();
        $input['user_id'] = Auth::user()->id;
        $result = $this->noticeRepo->create($input);
        event(new NoticeCreated($result[1]));
        session()->put('notice-button-click', true);
        if ($result) {
            $this->removeCached();
            Alert::success(trans("Notice::notice.alert.create_success"), 'success');
            return redirect()->route('notice.list');
        }
    }

    public function edit(Notice $notice)
    {
        return view("Notice::noticeEdit")->withNotice($notice);
    }

    public function update(Notice $notice, NoticeRequest $request)
    {
        $result = $this->noticeRepo->update($notice, $request->all());
        event(new NoticeUpdated($result[1]));
        if ($result) {
            $this->removeCached();
            Alert::success(trans("Notice::notice.alert.update_success"), 'success');
            return redirect()->route('notice.list');
        }
    }

    public function delete(Notice $notice)
    {
        if($notice->user_id == Auth::user()->id) {
            $this->noticeRepo->delete($notice->id);
            event(new NoticeDeleted($notice));
            $this->removeCached();
            Alert::success(trans("Notice::notice.alert.delete_success"), 'success');
            return redirect()->route('notice.list');
        } else {
            $this->removeCached();
            Alert::error(trans("Notice::notice.alert.error_message"), 'Error');
            return redirect()->route('notice.list');
        }
    }

    public function show(Notice $notice)
    {
        if($notice != null) {
            NoticeAlert::updateOrInsert(
                ['user_id' => auth()->user()->id, 'notice_id' => $notice->id],
                ['seen' => 1]
            );
            $this->removeCached();
            return view("Notice::noticeView")->withNotice($notice);
        } else {
            abort(404);
        }

    }

    public function removeCached()
    {
        Cache::forget('notice');
        Cache::forget('notice_alerts_all');
        Cache::forget('notice_alerts_seen');
    }


}
