<?php
/*
| -------------------------------------------------------------
| AUTHOR:           INILABS TEAM
| -------------------------------------------------------------
| EMAIL:            info@inilabs.net or labs.ini@gmail.com
| -------------------------------------------------------------
| WEBSITE:          http://inilabs.net
| -------------------------------------------------------------
*/
namespace Modules\Quotation\Controllers;

use App\Http\Controllers\Controller;
use Barryvdh\DomPDF\Facade as PDF;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Modules\Client\Models\Client;
use Modules\Currency\Models\Currency;
use Modules\Quotation\Models\Payment;
use Modules\Iauth\Models\Access\User\User;
use Modules\Iauth\Repositories\Access\User\UserRepositoryContract;
use Modules\Quotation\Models\Quotation;
use Modules\Quotation\Models\QuotationItem;
use Modules\Quotation\Repositories\QuotationRepository;
use Modules\Quotation\Repositories\QuotationItemRepository;
use Modules\Quotation\Requests\QuotationRequest;
use Modules\Quotation\Requests\PaymentRequest;
use Modules\Quotation\Requests\PaymentSettingsRequest;
use Modules\Product\Models\Product;
use Modules\Settings\Models\Settings;
use Modules\Tax\Models\Tax;
use Yajra\Datatables\Facades\Datatables;
use Alert, Cache;
use Omnipay\Omnipay;
use Illuminate\Support\Facades\Session;

class QuotationController extends Controller
{
    protected $quotationRepo;
    protected $userRepo;
    protected $quotationItemRepo;


    function __construct(QuotationRepository $quotation, UserRepositoryContract $client, QuotationItemRepository $item)
    {
        $this->quotationRepo = $quotation;
        $this->userRepo = $client;
        $this->quotationItemRepo = $item;
    }

    public function index()
    {
        return view("Quotation::quotationList");
    }

    public function quotationDataTable()
    {
        if(auth()->user()->hasrole(1)) {
            $quotations = $this->quotationRepo->orderBy('created_at', 'desc')->findAll();
        } else {
            $quotations = $this->quotationRepo->orderBy('created_at', 'desc')->findWhere(['client_id', '=', auth()->user()->id]);
        }

        return DataTables::of($quotations)
            ->editColumn('serial', function ($quotation){
                return "<a style='border: 1px solid; padding:2px;' href='".route('quotation.view', $quotation->id)."'>".$quotation->serial."</a>";
            })
            ->addColumn('client', function ($quotation){
                  return $quotation->client->name;
            })
            ->editColumn('grand_total', function ($quotation){
                return $quotation->currency->currency_symbol ? $quotation->currency->currency_symbol.number_format($quotation->grand_total, 2) : isite()->siteCurrencySymbol().number_format($quotation->grand_total, 2);
            })
            ->addColumn('actions', function ($quotation){
                return $quotation->action_buttons;
            })
            ->make(true);
    }


    public function add()
    {
        $client   = Cache::remember('all_clients',60, function () {
            return $this->userRepo->getAllUsersByRole('Client')->pluck('username', 'id')->toArray();
        });

        $taxes    = Cache::remember('taxes',60, function () {
            return Tax::all()->pluck('title', 'id')->toArray();
        });
        $currencies    = Cache::remember('currencies',60, function () {
            return Currency::all()->pluck('currency_code', 'id')->toArray();
        });
        $products = Cache::remember('products',60, function () {
            return Product::all()->pluck('name', 'amount')->toArray();
        });
        return view("Quotation::quotationCreate")->withClients($client)->withTaxes($taxes)->withProducts($products)->withCurrencies($currencies);
    }
    public function store(QuotationRequest $request)
    {

        if ($request->tax_id == "") {
            unset($request['tax_id']);
        }
        $products = collect($request->products)->transform(function($product) {
            $product['total'] = $product['quantity'] * $product['price'];
            return new QuotationItem($product);
        });

        if($products->isEmpty()) {
            return response()
                ->json([
                    'products_empty' => ['One or more Product is required.']
                ], 422);
        }

        $data = $request->except('products', 'action');
        $data['sub_total'] = $products->sum('total');
        $tax_amount = 0;
        if ($request->has('tax_id')) {
            $tax = Tax::findOrFail($request->tax_id);
            if ($tax->type==1) {
                $percentage = $tax->rate;
                $subTotal = $data['sub_total'];
                $tax_amount = ($percentage / 100) * $subTotal;
            } elseif($tax->type==2) {
                $tax_amount = $tax->rate;
            }
        }
        $data['grand_total'] = ($data['sub_total'] - $data['discount']) + $tax_amount;
        $data['user_id'] = auth()->user()->id;

        $quotation = Quotation::create($data);

        $quotation->products()->saveMany($products);

        $this->removeCached();

        return response()
            ->json([
                'created' => true,
                'id' => $quotation->id
            ]);
    }

    public function edit($id)
    {
        $client         = Cache::remember('all_clients',60, function () {
            return $this->userRepo->getAllUsersByRole('Client')->pluck('username', 'id')->toArray();
        });

        $taxes    = Cache::remember('taxes',60, function () {
            return Tax::all()->pluck('title', 'id')->toArray();
        });

        $currencies    = Cache::remember('currencies',60, function () {
            return Currency::all()->pluck('currency_code', 'id')->toArray();
        });

        $products = Cache::remember('products',60, function () {
            return Product::all()->pluck('name', 'amount')->toArray();
        });
        $quotation = Quotation::with('products')->findOrFail($id);
        return view('Quotation::quotationEdit')->withClients($client)->withTaxes($taxes)->withQuotation($quotation)->withProducts($products)->withCurrencies($currencies);
    }

    public function update(QuotationRequest $request, $id)
    {

        $quotation = Quotation::findOrFail($id);

        $products = collect($request->products)->transform(function($product) {
            $product['total'] = $product['quantity'] * $product['price'];
            return new QuotationItem($product);
        });

        if($products->isEmpty()) {
            return response()
                ->json([
                    'products_empty' => ['One or more Product is required.']
                ], 422);
        }

        $data = $request->except('products', 'action');
        $data['sub_total'] = $products->sum('total');
        $tax_amount = 0;
        if ($request->has('tax_id')) {
            $tax = Tax::findOrFail($request->tax_id);
            if ($tax->type==1) {
                $percentage = $tax->rate;
                $subTotal = $data['sub_total'];
                $tax_amount = ($percentage / 100) * $subTotal;
            } elseif($tax->type==2) {
                $tax_amount = $tax->rate;
            }
        }
        $data['grand_total'] = ($data['sub_total'] - $data['discount']) + $tax_amount;

        $quotation->update($data);

        QuotationItem::where('quotation_id', $quotation->id)->delete();

        $quotation->products()->saveMany($products);

        $this->removeCached();

        return response()
            ->json([
                'updated' => true,
                'id' => $quotation->id,
                'url' => route('quotation.view', $quotation->id)
            ]);
    }
    public function view($id)
    {
        $quotation   = Cache::remember('quotation',60, function () use ($id) {
            return Quotation::with('products')->findOrFail($id);
        });

        $tax = Tax::find($quotation->tax_id);
        $client = User::findOrFail($quotation->client_id);
        $relationTableData = $client->hasOneRelation('Modules\Client\Models\Client')->first();
        return view("Quotation::quotationView")->withQuotation($quotation)->withTax($tax)->withClient($client)->withCompany($relationTableData);
    }

    public function delete($id)
    {
        $quotation = Quotation::findOrFail($id);
        QuotationItem::where('quotation_id', $quotation->id)
            ->delete();
        $quotation->delete();
        $this->removeCached();
        return redirect()
            ->route('quotation.list');
    }


    //send quotation email
    public function sendEmail($id, Request $request)
    {
        if(getenv('MAIL_USERNAME')!=null && getenv('MAIL_PASSWORD') != null) {
            $this->validate($request, [
                'email' => 'required|email',
                'subject' => 'required',
                'body' => 'required',
            ]);
            $attachedFileName = '';
            $title      = "<h2>$request->subject</h2>";
            $content    = "<p>$request->body</p>";
            if ($request->has('attachStatus')) {
                $attachedFileName = $this->storeQuotation($id);
            }
            Mail::send('emails.send', ['title' => $title, 'content' => $content], function ($message) use ($request, $attachedFileName)
            {
                $message->from(isite()->siteEmail(), isite()->siteName());
                $message->to($request->email);
                if ($attachedFileName!="") {
                    $message->attach($attachedFileName, ['mime' => 'pdf']);
                }
            });
            $data = [
                'success' => true,
            ];
            return response()->json($data);
        } else {
            $data = [
                'error' => "Please Set Smtp Information!",
            ];
            return response()->json($data);
        }
    }

    public function removeCached()
    {
        Cache::forget('quotation');
    }



    public function quotationPdf($id)
    {
        $quotation   = Cache::remember('quotation',60, function () use ($id) {
            return Quotation::with('products')->findOrFail($id);
        });
        $tax = Tax::find($quotation->tax_id);
        $client = User::findOrFail($quotation->client_id);
        $company = $client->hasOneRelation('Modules\Client\Models\Client')->first();

        $pdf = PDF::loadView('Quotation::quotationPDF', compact('quotation','tax', 'client', 'company'));
        return $pdf->stream();
    }

    public function storeQuotation($id)
    {
        $quotation   = Cache::remember('quotation',60, function () use ($id) {
            return Quotation::with('products')->findOrFail($id);
        });
        $tax = Tax::find($quotation->tax_id);
        $client = User::findOrFail($quotation->client_id);
        $company = $client->hasOneRelation('Modules\Client\Models\Client')->first();

        $pdf = PDF::loadView('Quotation::quotationPDF', compact('quotation','tax', 'client', 'company'));

        $quotationPath = public_path('assets/pdf/').date('Y-m-d-H-i-s')."-quote-".$quotation->serial.'.pdf';
        if($pdf->save($quotationPath)) {
            return $quotationPath;
        } else {
            return false;
        }
    }


}
