<?php
/*
| -------------------------------------------------------------
| iBill - Simplest and Best Billing & Invoice Manager
| -------------------------------------------------------------
| AUTHOR:           INILABS TEAM
| -------------------------------------------------------------
| EMAIL:            info@inilabs.net
| -------------------------------------------------------------
| COPYRIGHT:        RESERVED BY INILABS IT
| -------------------------------------------------------------
| WEBSITE:          http://inilabs.net
| -------------------------------------------------------------
*/
namespace Modules\Todo\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Modules\Todo\Events\History\TodoCreated;
use Modules\Todo\Events\History\TodoDeleted;
use Modules\Todo\Events\History\TodoUpdated;
use Modules\Todo\Models\Todo;
use Modules\Todo\Repositories\TodoRepository;
use Modules\Todo\Requests\TodoRequest;
use Yajra\Datatables\Facades\Datatables;
use Alert, Cache;
use Carbon\Carbon;

class TodoController extends Controller
{
    protected $todoRepo;

    function __construct(TodoRepository $todo)
    {
        $this->todoRepo = $todo;
    }

    public function langUses()
    {
        return view("Todo::index");
    }

    public function index()
    {
        return view("Todo::todoList");
    }

    public function todoDataTable()
    {
        return DataTables::of($this->todoRepo->findWhere(['user_id', '=', Auth::user()->id]))
            ->addColumn('actions', function ($todo){
                return $todo->action_buttons;
            })
            ->addColumn('status', function ($todo){
                return $todo->status_button;
            })
            ->editColumn('created_at', function ($todo) {
                return $todo->created_at ? with(new Carbon($todo->created_at))->addSeconds(5)->diffForHumans() : '';
            })
            ->make(true);
    }

    public function add()
    {
        return view("Todo::todoCreate");
    }

    public function create(TodoRequest $request)
    {
        $input = $request->all();
        $input['user_id'] = Auth::user()->id;
        $result = $this->todoRepo->create($input);

        if ($result) {
            event(new TodoCreated($result[1]));
            $this->removeCached();
            Alert::success(trans("Todo::todo.alert.create_success"), 'success');
        }
        return redirect()->route('todo.list');
    }

    public function edit(Todo $id)
    {
        $todo = $id;
        if($todo->user_id != Auth::user()->id) {
            abort(404);
        }
        return view("Todo::todoEdit")->withTodo($todo);
    }

    public function update(Todo $id, TodoRequest $request)
    {
        $result = $this->todoRepo->update($id, $request->all());

        if ($result) {
            event(new TodoUpdated($result[1]));
            $this->removeCached();
            Alert::success(trans("Todo::todo.alert.update_success"), 'success');
        }
        return redirect()->route('todo.list');
    }

    public function delete(Todo $id)
    {
        $todo = $id;
        if($todo->user_id == Auth::user()->id) {
            $this->todoRepo->delete($todo->id);
            event(new TodoDeleted($todo));
            $this->removeCached();
            Alert::success(trans("Todo::todo.alert.delete_success"), 'success');
        } else {
            Alert::error(trans("Todo::todo.alert.auth_error"), 'error');
        }
        return redirect()->route('todo.list');
    }

    public function status(Todo $id)
    {
        $todo = $id;
        $data = [];
        if($todo->user_id == Auth::user()->id) {
            if ($todo->status==1) {
                $data['status'] = 0;
            } else {
                $data['status'] = 1;
            }
            $this->todoRepo->update($todo->id, $data);
            $this->removeCached();
            Alert::success(trans("Todo::todo.alert.update_success"), 'success');
        } else {
            Alert::error(trans("Todo::todo.alert.auth_error"), 'error');
        }
        return redirect()->route('todo.list');
    }

    public function removeCached()
    {
        Cache::forget('user_todo('.auth()->user()->id.')');
    }

}
